/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.sql.engine.exec.memory.structures.file;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.ignite3.internal.binarytuple.BinaryTupleParser;
import org.apache.ignite3.internal.fileio.FileIo;
import org.apache.ignite3.internal.fileio.FileIoFactory;
import org.apache.ignite3.internal.lang.IgniteStringFormatter;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.schema.BinaryTuple;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.file.DataDirectory;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.file.ExternalCollectionUtils;
import org.apache.ignite3.internal.util.IgniteUtils;
import org.apache.ignite3.sql.SqlException;
import org.gridgain.lang.GridgainErrorGroups;
import org.jetbrains.annotations.TestOnly;

public class ExternalFileStore
implements AutoCloseable,
Iterable<BinaryTuple> {
    private static final IgniteLogger log = Loggers.forClass(ExternalFileStore.class);
    static final int ROW_HEADER_SIZE = 5;
    private static final int FLAGS_OFFSET = 4;
    private static final byte TOMBSTONE_FLAG = 1;
    private static final byte[] TOMBSTONE_BYTES = new byte[]{1};
    private final ByteBuffer headerBuff = ByteBuffer.allocate(5).order(BinaryTupleParser.ORDER);
    private final Path file;
    private final FileIo fileIo;
    private final int columnsCount;
    private long lastWrittenOffset;
    private int size;
    private boolean isClosed = false;
    private long headAddr = 0L;

    private static boolean hasTombstoneFlag(byte flags) {
        return (flags & 1) != 0;
    }

    public <RowT> ExternalFileStore(FileIoFactory fileIoFactory, DataDirectory workDir, int columnsCount) {
        this.columnsCount = columnsCount;
        this.file = workDir.createFile("data");
        try {
            this.fileIo = fileIoFactory.create(this.file, StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        catch (Exception e) {
            throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Failed to create spill file.", (Throwable)e);
        }
        if (log.isDebugEnabled()) {
            log.debug("Created spill file " + this.file.getFileName(), new Object[0]);
        }
    }

    public synchronized boolean contains(long address) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        return !ExternalFileStore.hasTombstoneFlag(header.get(4));
    }

    @TestOnly
    public long lastWrittenOffset() {
        return this.lastWrittenOffset;
    }

    @Nullable
    public synchronized BinaryTuple read(long address) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        int size = header.getInt();
        byte flags = header.get();
        if (ExternalFileStore.hasTombstoneFlag(flags)) {
            this.setFilePosition(address + (long)size + 5L);
            return null;
        }
        ByteBuffer readBuff = ByteBuffer.allocate(size).order(BinaryTupleParser.ORDER);
        readBuff.clear();
        readBuff.limit(size);
        this.readFromFile(readBuff);
        readBuff.flip();
        return new BinaryTuple(this.columnsCount, readBuff);
    }

    public synchronized int read(long address, RowReader reader) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        int size = header.getInt();
        byte flags = header.get();
        if (ExternalFileStore.hasTombstoneFlag(flags)) {
            this.setFilePosition(address + (long)size + 5L);
            return 0;
        }
        reader.readRow(size, this::readFromFile);
        return size;
    }

    public synchronized int read(long address, ByteBuffer buff) {
        return this.read(address, (int rowSize, Consumer<ByteBuffer> io) -> io.accept(buff));
    }

    public synchronized void update(long address, ByteBuffer rowBuff) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        int size = header.getInt();
        byte flags = header.get();
        if (ExternalFileStore.hasTombstoneFlag(flags)) {
            String message = IgniteStringFormatter.format("Attempted to update a tombstone entry. Address: {}.", address);
            throw new IllegalArgumentException(message);
        }
        assert (rowBuff.order() == BinaryTupleParser.ORDER) : "Incorrect byte order";
        int newRowSize = rowBuff.remaining();
        if (newRowSize > size) {
            String message = IgniteStringFormatter.format("Update buffer size > record size. Expected {} but got {}.", size, newRowSize);
            throw new IllegalArgumentException(message);
        }
        ByteBuffer rowHeader = this.prepareHeader(size);
        this.doWriteRow(address, rowHeader, rowBuff, false);
    }

    public synchronized void update(long address, RowWriter rowWriter) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        int size = header.getInt();
        byte flags = header.get();
        if (ExternalFileStore.hasTombstoneFlag(flags)) {
            String message = IgniteStringFormatter.format("Attempted to update a tombstone entry. Address: {}.", address);
            throw new IllegalArgumentException(message);
        }
        rowWriter.write((ByteBuffer byteBuffer) -> {
            if (byteBuffer.remaining() > size) {
                throw new IllegalStateException(IgniteStringFormatter.format("Failed to update row in-place. Address: {}.", address));
            }
            this.doWriteRow(address + 5L, header, (ByteBuffer)byteBuffer, false);
        });
    }

    public synchronized long write(ByteBuffer rowBuff) {
        this.checkClosed();
        int rowSize = rowBuff.remaining();
        ByteBuffer rowHeader = this.prepareHeader(rowSize);
        return this.doWriteRow(this.lastWrittenOffset, rowHeader, rowBuff, true);
    }

    public synchronized long write(BinaryTuple binaryTuple) {
        this.checkClosed();
        assert (binaryTuple.elementCount() == this.columnsCount);
        return this.writeRow(binaryTuple);
    }

    public synchronized void remove(long address) {
        assert (address >= 0L);
        assert (address + 5L < this.lastWrittenOffset);
        this.checkClosed();
        ByteBuffer header = this.readHeader(address);
        if (ExternalFileStore.hasTombstoneFlag(header.get(4))) {
            return;
        }
        try {
            this.fileIo.position(address + 4L);
            this.fileIo.write(TOMBSTONE_BYTES, 0, 1);
            --this.size;
            if (this.headAddr == address) {
                this.headAddr = this.findNext(address + (long)header.getInt() + 5L, this.lastWrittenOffset);
            }
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    @Override
    public synchronized Iterator<BinaryTuple> iterator() {
        this.checkClosed();
        return new BinaryTupleChunkIterator(this.headAddr, this.lastWrittenOffset);
    }

    public synchronized Iterator<ByteBuffer> rowIterator(RowReader rowReader, ByteBuffer buffer) {
        return new RowChunkIterator(this.headAddr, this.lastWrittenOffset, rowReader, buffer);
    }

    public synchronized int size() {
        this.checkClosed();
        return this.size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reset() {
        ExternalFileStore externalFileStore = this;
        synchronized (externalFileStore) {
            this.checkClosed();
            this.lastWrittenOffset = 0L;
            this.headAddr = 0L;
            this.size = 0;
            try {
                this.fileIo.clear();
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("External row store file cleaned: " + this.file.getFileName(), new Object[0]);
        }
    }

    @Override
    public void close() {
        this.close0();
    }

    @Nullable
    synchronized BinaryTuple readFirst() {
        this.checkClosed();
        if (this.headAddr < this.lastWrittenOffset) {
            return this.read(this.headAddr);
        }
        return null;
    }

    @Nullable
    synchronized BinaryTuple removeFirst() {
        BinaryTuple tuple;
        this.checkClosed();
        if (this.headAddr < this.lastWrittenOffset && (tuple = this.read(this.headAddr)) != null) {
            this.remove(this.headAddr);
            return tuple;
        }
        return null;
    }

    private void readFromFile(ByteBuffer buff) {
        try {
            this.fileIo.read(buff);
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private long writeRow(BinaryTuple binaryTuple) {
        assert (binaryTuple != null);
        ByteBuffer rowBody = binaryTuple.byteBuffer();
        ByteBuffer rowHeader = this.prepareHeader(rowBody.limit());
        return this.doWriteRow(this.lastWrittenOffset, rowHeader, rowBody, true);
    }

    private long doWriteRow(long rowAddress, ByteBuffer rowHeader, ByteBuffer rowBody, boolean newRow) {
        try {
            this.fileIo.position(rowAddress);
            this.fileIo.writeFully(rowHeader);
            this.fileIo.writeFully(rowBody);
            if (newRow) {
                ++this.size;
                this.lastWrittenOffset = this.fileIo.position();
            }
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
        return rowAddress;
    }

    private synchronized long currentFilePosition() {
        try {
            return this.fileIo.position();
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void setFilePosition(long pos) {
        try {
            this.fileIo.position(pos);
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void checkClosed() {
        if (this.isClosed) {
            throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Row store has been closed.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void close0() {
        ExternalFileStore externalFileStore = this;
        synchronized (externalFileStore) {
            if (this.isClosed) {
                return;
            }
            this.isClosed = true;
        }
        IgniteUtils.closeQuiet(this.fileIo);
        if (!this.file.toFile().delete()) {
            log.info("Failed to remove spill file " + this.file.getFileName(), new Object[0]);
        } else if (log.isDebugEnabled()) {
            log.debug("Spill file removed " + this.file.getFileName(), new Object[0]);
        }
    }

    private ByteBuffer readHeader(long offset) {
        this.headerBuff.clear();
        try {
            this.fileIo.position(offset);
            this.fileIo.readFully(this.headerBuff);
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
        this.headerBuff.flip();
        return this.headerBuff;
    }

    private ByteBuffer prepareHeader(int rowSize) {
        this.headerBuff.clear();
        this.headerBuff.putInt(rowSize);
        this.headerBuff.put((byte)0);
        this.headerBuff.flip();
        return this.headerBuff;
    }

    private synchronized long findNext(long start, long end) {
        this.checkClosed();
        while (start < end) {
            ByteBuffer header = this.readHeader(start);
            int rowSize = header.getInt();
            boolean isTombstone = ExternalFileStore.hasTombstoneFlag(header.get());
            if (!isTombstone) {
                return start;
            }
            start += (long)(rowSize + 5);
        }
        return start;
    }

    @FunctionalInterface
    public static interface RowReader {
        public void readRow(int var1, Consumer<ByteBuffer> var2);
    }

    @FunctionalInterface
    public static interface RowWriter {
        public void write(Consumer<ByteBuffer> var1);
    }

    private class BinaryTupleChunkIterator
    extends BaseChuckIterator<BinaryTuple> {
        BinaryTupleChunkIterator(long start, long end) {
            super(start, end);
        }

        @Override
        BinaryTuple nextValue() {
            BinaryTuple binaryTuple = ExternalFileStore.this.read(this.curPos);
            assert (binaryTuple != null);
            return binaryTuple;
        }
    }

    private class RowChunkIterator
    extends BaseChuckIterator<ByteBuffer> {
        private final RowReader rowReader;
        private final ByteBuffer buffer;

        RowChunkIterator(long start, long end, RowReader rowReader, ByteBuffer buffer) {
            super(start, end);
            this.rowReader = rowReader;
            this.buffer = buffer;
        }

        @Override
        ByteBuffer nextValue() {
            int r = ExternalFileStore.this.read(this.curPos, this.rowReader);
            assert (r > 0);
            return this.buffer;
        }
    }

    private abstract class BaseChuckIterator<T>
    implements Iterator<T> {
        private final long end;
        long curPos;

        BaseChuckIterator(long start, long end) {
            this.curPos = start;
            this.end = end;
        }

        @Override
        public final boolean hasNext() {
            this.curPos = ExternalFileStore.this.findNext(this.curPos, this.end);
            return this.curPos < this.end;
        }

        @Override
        public final T next() {
            this.curPos = ExternalFileStore.this.findNext(this.curPos, this.end);
            if (this.curPos < this.end) {
                T nextValue = this.nextValue();
                this.curPos = ExternalFileStore.this.currentFilePosition();
                return nextValue;
            }
            throw new NoSuchElementException();
        }

        abstract T nextValue();
    }
}

