/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.raft.storage.segstore;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.ignite3.internal.close.ManuallyCloseable;
import org.apache.ignite3.internal.failure.FailureProcessor;
import org.apache.ignite3.internal.lang.IgniteInternalException;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.raft.configuration.LogStorageConfiguration;
import org.apache.ignite3.internal.raft.configuration.LogStorageConfigurationSchema;
import org.apache.ignite3.internal.raft.configuration.LogStorageView;
import org.apache.ignite3.internal.raft.configuration.RaftConfiguration;
import org.apache.ignite3.internal.raft.storage.segstore.EntrySearchResult;
import org.apache.ignite3.internal.raft.storage.segstore.IndexFileManager;
import org.apache.ignite3.internal.raft.storage.segstore.IndexMemTable;
import org.apache.ignite3.internal.raft.storage.segstore.RaftLogCheckpointer;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentFile;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentFilePointer;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentFileWithMemtable;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentInfo;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentPayload;
import org.apache.ignite3.internal.raft.storage.segstore.SegmentPayloadParser;
import org.apache.ignite3.internal.raft.storage.segstore.WriteBufferWithMemtable;
import org.apache.ignite3.internal.raft.storage.segstore.WriteModeIndexMemTable;
import org.apache.ignite3.lang.ErrorGroups;
import org.apache.ignite3.raft.jraft.entity.LogEntry;
import org.apache.ignite3.raft.jraft.entity.codec.LogEntryDecoder;
import org.apache.ignite3.raft.jraft.entity.codec.LogEntryEncoder;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

class SegmentFileManager
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(SegmentFileManager.class);
    private static final int ROLLOVER_WAIT_TIMEOUT_MS = 30000;
    static final int MAGIC_NUMBER = 1457567014;
    static final int FORMAT_VERSION = 1;
    private static final String SEGMENT_FILE_NAME_FORMAT = "segment-%010d-%010d.bin";
    private static final Pattern SEGMENT_FILE_NAME_PATTERN = Pattern.compile("segment-(?<ordinal>\\d{10})-(?<generation>\\d{10})\\.bin");
    static final byte[] HEADER_RECORD = ByteBuffer.allocate(8).order(SegmentFile.BYTE_ORDER).putInt(1457567014).putInt(1).array();
    static final byte[] SWITCH_SEGMENT_RECORD = new byte[8];
    private final Path segmentFilesDir;
    private final int stripes;
    private final AtomicReference<SegmentFileWithMemtable> currentSegmentFile = new AtomicReference();
    private final RaftLogCheckpointer checkpointer;
    private final IndexFileManager indexFileManager;
    private final int segmentFileSize;
    private final int maxLogEntrySize;
    private final boolean isSync;
    private final Object rolloverLock = new Object();
    private volatile int curSegmentFileOrdinal;
    private boolean isStopped;

    SegmentFileManager(String nodeName, Path baseDir, int stripes, FailureProcessor failureProcessor, RaftConfiguration raftConfiguration, LogStorageConfiguration storageConfiguration) throws IOException {
        this.segmentFilesDir = baseDir.resolve("segments");
        this.stripes = stripes;
        this.isSync = (Boolean)raftConfiguration.fsync().value();
        Files.createDirectories(this.segmentFilesDir, new FileAttribute[0]);
        LogStorageView logStorageView = (LogStorageView)storageConfiguration.value();
        this.segmentFileSize = Math.toIntExact(logStorageView.segmentFileSizeBytes());
        this.maxLogEntrySize = SegmentFileManager.maxLogEntrySize(logStorageView);
        this.indexFileManager = new IndexFileManager(baseDir);
        this.checkpointer = new RaftLogCheckpointer(nodeName, this.indexFileManager, failureProcessor, logStorageView.maxCheckpointQueueSize());
    }

    void start() throws IOException {
        LOG.info("Starting segment file manager [segmentFilesDir={}, fileSize={}].", this.segmentFilesDir, this.segmentFileSize);
        this.indexFileManager.cleanupTmpFiles();
        SegmentPayloadParser payloadParser = new SegmentPayloadParser(this.stripes);
        Path lastSegmentFilePath = null;
        try (Stream<Path> segmentFiles = Files.list(this.segmentFilesDir);){
            Iterator it = segmentFiles.sorted().iterator();
            while (it.hasNext()) {
                Path segmentFilePath = (Path)it.next();
                if (!it.hasNext()) {
                    lastSegmentFilePath = segmentFilePath;
                    continue;
                }
                int segmentFileOrdinal = SegmentFileManager.segmentFileOrdinal(segmentFilePath);
                if (this.indexFileManager.indexFileExists(segmentFileOrdinal)) continue;
                LOG.info("Creating missing index file for segment file {}.", segmentFilePath);
                SegmentFileWithMemtable segmentFileWithMemtable = this.recoverSegmentFile(segmentFilePath, payloadParser);
                this.indexFileManager.recoverIndexFile(segmentFileWithMemtable.memtable().transitionToReadMode(), segmentFileOrdinal);
            }
        }
        if (lastSegmentFilePath == null) {
            this.currentSegmentFile.set(this.allocateNewSegmentFile(0));
        } else {
            this.curSegmentFileOrdinal = SegmentFileManager.segmentFileOrdinal(lastSegmentFilePath);
            this.currentSegmentFile.set(this.recoverLatestSegmentFile(lastSegmentFilePath, payloadParser));
        }
        LOG.info("Segment file manager recovery completed. Current segment file: {}.", lastSegmentFilePath);
        this.indexFileManager.start();
        this.checkpointer.start();
    }

    Path segmentFilesDir() {
        return this.segmentFilesDir;
    }

    Path indexFilesDir() {
        return this.indexFileManager.indexFilesDir();
    }

    @TestOnly
    IndexFileManager indexFileManager() {
        return this.indexFileManager;
    }

    private SegmentFileWithMemtable allocateNewSegmentFile(int fileOrdinal) throws IOException {
        Path path = this.segmentFilesDir.resolve(SegmentFileManager.segmentFileName(fileOrdinal, 0));
        SegmentFile segmentFile = SegmentFile.createNew(path, this.segmentFileSize, this.isSync);
        SegmentFileManager.writeHeader(segmentFile);
        return new SegmentFileWithMemtable(segmentFile, new IndexMemTable(this.stripes), false);
    }

    private SegmentFileWithMemtable recoverLatestSegmentFile(Path segmentFilePath, SegmentPayloadParser payloadParser) throws IOException {
        SegmentFile segmentFile = SegmentFile.openExisting(segmentFilePath, this.isSync);
        WriteModeIndexMemTable memTable = payloadParser.recoverMemtable(segmentFile, segmentFilePath, true);
        return new SegmentFileWithMemtable(segmentFile, memTable, false);
    }

    private SegmentFileWithMemtable recoverSegmentFile(Path segmentFilePath, SegmentPayloadParser payloadParser) throws IOException {
        SegmentFile segmentFile = SegmentFile.openExisting(segmentFilePath, this.isSync);
        WriteModeIndexMemTable memTable = payloadParser.recoverMemtable(segmentFile, segmentFilePath, false);
        return new SegmentFileWithMemtable(segmentFile, memTable, false);
    }

    private static String segmentFileName(int fileOrdinal, int generation) {
        return String.format(SEGMENT_FILE_NAME_FORMAT, fileOrdinal, generation);
    }

    private static SegmentFileWithMemtable convertToReadOnly(SegmentFileWithMemtable segmentFile) {
        return new SegmentFileWithMemtable(segmentFile.segmentFile(), segmentFile.memtable(), true);
    }

    void appendEntry(long groupId, LogEntry entry, LogEntryEncoder encoder) throws IOException {
        int segmentEntrySize = SegmentPayload.size(entry, encoder);
        if (segmentEntrySize > this.maxLogEntrySize) {
            throw new IllegalArgumentException(String.format("Segment entry is too big (%d bytes), maximum allowed segment entry size: %d bytes.", segmentEntrySize, this.maxLogEntrySize));
        }
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(segmentEntrySize);){
            ByteBuffer segmentBuffer = writeBufferWithMemtable.buffer();
            int segmentOffset = segmentBuffer.position();
            SegmentPayload.writeTo(segmentBuffer, groupId, segmentEntrySize, entry, encoder);
            writeBufferWithMemtable.memtable().appendSegmentFileOffset(groupId, entry.getId().getIndex(), segmentOffset);
        }
    }

    @Nullable
    LogEntry getEntry(long groupId, long logIndex, LogEntryDecoder decoder) throws IOException {
        ByteBuffer entryBuffer = this.getEntry(groupId, logIndex);
        return entryBuffer == null ? null : SegmentPayload.readFrom(entryBuffer, decoder);
    }

    @Nullable
    private ByteBuffer getEntry(long groupId, long logIndex) throws IOException {
        EntrySearchResult searchResult = this.getEntryFromCurrentMemtable(groupId, logIndex);
        if (searchResult.searchOutcome() == EntrySearchResult.SearchOutcome.CONTINUE_SEARCH && (searchResult = this.checkpointer.findSegmentPayloadInQueue(groupId, logIndex)).searchOutcome() == EntrySearchResult.SearchOutcome.CONTINUE_SEARCH) {
            searchResult = this.readFromOtherSegmentFiles(groupId, logIndex);
        }
        switch (searchResult.searchOutcome()) {
            case SUCCESS: {
                return searchResult.entryBuffer();
            }
            case NOT_FOUND: {
                return null;
            }
        }
        throw new IllegalStateException("Unexpected search outcome: " + searchResult.searchOutcome());
    }

    private EntrySearchResult getEntryFromCurrentMemtable(long groupId, long logIndex) {
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        if (segmentInfo == null) {
            return EntrySearchResult.continueSearch();
        }
        if (logIndex >= segmentInfo.lastLogIndexExclusive()) {
            return EntrySearchResult.notFound();
        }
        if (logIndex < segmentInfo.firstIndexKept()) {
            return EntrySearchResult.notFound();
        }
        int segmentPayloadOffset = segmentInfo.getOffset(logIndex);
        if (segmentPayloadOffset == SegmentInfo.MISSING_SEGMENT_FILE_OFFSET) {
            return EntrySearchResult.continueSearch();
        }
        ByteBuffer entryBuffer = currentSegmentFile.segmentFile().buffer().position(segmentPayloadOffset);
        return EntrySearchResult.success(entryBuffer);
    }

    void truncateSuffix(long groupId, long lastLogIndexKept) throws IOException {
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(24);){
            SegmentPayload.writeTruncateSuffixRecordTo(writeBufferWithMemtable.buffer(), groupId, lastLogIndexKept);
            writeBufferWithMemtable.memtable().truncateSuffix(groupId, lastLogIndexKept);
        }
    }

    void truncatePrefix(long groupId, long firstLogIndexKept) throws IOException {
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(24);){
            SegmentPayload.writeTruncatePrefixRecordTo(writeBufferWithMemtable.buffer(), groupId, firstLogIndexKept);
            writeBufferWithMemtable.memtable().truncatePrefix(groupId, firstLogIndexKept);
        }
    }

    void reset(long groupId, long nextLogIndex) throws IOException {
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(24);){
            SegmentPayload.writeResetRecordTo(writeBufferWithMemtable.buffer(), groupId, nextLogIndex);
            writeBufferWithMemtable.memtable().reset(groupId, nextLogIndex);
        }
    }

    private WriteBufferWithMemtable reserveBytesWithRollover(int size) throws IOException {
        SegmentFileWithMemtable segmentFileWithMemtable;
        SegmentFile.WriteBuffer writeBuffer;
        while ((writeBuffer = (segmentFileWithMemtable = this.currentSegmentFile()).segmentFile().reserve(size)) == null) {
            this.initiateRollover(segmentFileWithMemtable);
        }
        return new WriteBufferWithMemtable(writeBuffer, segmentFileWithMemtable.memtable());
    }

    long firstLogIndexInclusiveOnRecovery(long groupId) {
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        if (segmentInfo != null && segmentInfo.firstIndexKept() != -1L) {
            return segmentInfo.firstIndexKept();
        }
        long firstLogIndexFromIndexStorage = this.indexFileManager.firstLogIndexInclusive(groupId);
        if (firstLogIndexFromIndexStorage != -1L) {
            return firstLogIndexFromIndexStorage;
        }
        return segmentInfo == null ? -1L : segmentInfo.firstLogIndexInclusive();
    }

    long lastLogIndexExclusiveOnRecovery(long groupId) {
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        if (segmentInfo != null) {
            return segmentInfo.lastLogIndexExclusive();
        }
        return this.indexFileManager.lastLogIndexExclusive(groupId);
    }

    private SegmentFileWithMemtable currentSegmentFile() {
        SegmentFileWithMemtable segmentFile = this.currentSegmentFile.get();
        if (!segmentFile.readOnly()) {
            return segmentFile;
        }
        try {
            Object object = this.rolloverLock;
            synchronized (object) {
                while (true) {
                    if (this.isStopped) {
                        throw new IgniteInternalException(ErrorGroups.Common.NODE_STOPPING_ERR);
                    }
                    segmentFile = this.currentSegmentFile.get();
                    if (!segmentFile.readOnly()) {
                        return segmentFile;
                    }
                    this.rolloverLock.wait(30000L);
                }
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, "Interrupted while waiting for rollover.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initiateRollover(SegmentFileWithMemtable observedSegmentFile) throws IOException {
        if (!this.currentSegmentFile.compareAndSet(observedSegmentFile, SegmentFileManager.convertToReadOnly(observedSegmentFile))) {
            return;
        }
        this.checkpointer.onRollover(observedSegmentFile.segmentFile(), observedSegmentFile.memtable().transitionToReadMode());
        Object object = this.rolloverLock;
        synchronized (object) {
            if (this.isStopped) {
                throw new IgniteInternalException(ErrorGroups.Common.NODE_STOPPING_ERR);
            }
            this.currentSegmentFile.set(this.allocateNewSegmentFile(++this.curSegmentFileOrdinal));
            this.rolloverLock.notifyAll();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws Exception {
        Object object = this.rolloverLock;
        synchronized (object) {
            if (this.isStopped) {
                return;
            }
            this.isStopped = true;
            SegmentFileWithMemtable segmentFile = this.currentSegmentFile.get();
            if (segmentFile != null) {
                segmentFile.segmentFile().close();
            }
            this.rolloverLock.notifyAll();
        }
        this.checkpointer.stop();
    }

    private static void writeHeader(SegmentFile segmentFile) {
        try (SegmentFile.WriteBuffer writeBuffer = segmentFile.reserve(HEADER_RECORD.length);){
            assert (writeBuffer != null);
            writeBuffer.buffer().put(HEADER_RECORD);
        }
    }

    private EntrySearchResult readFromOtherSegmentFiles(long groupId, long logIndex) throws IOException {
        SegmentFilePointer segmentFilePointer = this.indexFileManager.getSegmentFilePointer(groupId, logIndex);
        if (segmentFilePointer == null) {
            return EntrySearchResult.notFound();
        }
        Path path = this.segmentFilesDir.resolve(SegmentFileManager.segmentFileName(segmentFilePointer.fileOrdinal(), 0));
        SegmentFile segmentFile = SegmentFile.openExisting(path, this.isSync);
        ByteBuffer buffer = segmentFile.buffer().position(segmentFilePointer.payloadOffset());
        return EntrySearchResult.success(buffer);
    }

    private static int segmentFileOrdinal(Path segmentFile) {
        String fileName = segmentFile.getFileName().toString();
        Matcher matcher = SEGMENT_FILE_NAME_PATTERN.matcher(fileName);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid segment file name format: %s.", segmentFile));
        }
        return Integer.parseInt(matcher.group("ordinal"));
    }

    private static int maxLogEntrySize(LogStorageView storageConfiguration) {
        int valueFromConfig = storageConfiguration.maxLogEntrySizeBytes();
        if (valueFromConfig != -1) {
            return valueFromConfig;
        }
        return LogStorageConfigurationSchema.computeDefaultMaxLogEntrySizeBytes(storageConfiguration.segmentFileSizeBytes());
    }
}

