/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.partition.replicator.schemacompat;

import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.ignite3.internal.catalog.CatalogService;
import org.apache.ignite3.internal.catalog.descriptors.CatalogTableColumnDescriptor;
import org.apache.ignite3.internal.catalog.descriptors.CatalogTableDescriptor;
import org.apache.ignite3.internal.hlc.HybridTimestamp;
import org.apache.ignite3.internal.partition.replicator.schema.ColumnDefinitionDiff;
import org.apache.ignite3.internal.partition.replicator.schema.FullTableSchema;
import org.apache.ignite3.internal.partition.replicator.schema.TableDefinitionDiff;
import org.apache.ignite3.internal.partition.replicator.schema.ValidationSchemasSource;
import org.apache.ignite3.internal.partition.replicator.schemacompat.CompatValidationResult;
import org.apache.ignite3.internal.partition.replicator.schemacompat.IncompatibleSchemaVersionException;
import org.apache.ignite3.internal.partition.replicator.schemacompat.InternalSchemaVersionMismatchException;
import org.apache.ignite3.internal.partition.replicator.schemacompat.TableDefinitionDiffKey;
import org.apache.ignite3.internal.schema.SchemaSyncService;
import org.apache.ignite3.internal.tx.TransactionIds;
import org.jetbrains.annotations.Nullable;

public class SchemaCompatibilityValidator {
    private final ValidationSchemasSource validationSchemasSource;
    private final CatalogService catalogService;
    private final SchemaSyncService schemaSyncService;
    private final ConcurrentMap<TableDefinitionDiffKey, TableDefinitionDiff> diffCache = new ConcurrentHashMap<TableDefinitionDiffKey, TableDefinitionDiff>();
    private static final List<ForwardCompatibilityValidator> FORWARD_COMPATIBILITY_VALIDATORS = List.of(new RenameTableValidator(), new AddColumnsValidator(), new DropColumnsValidator(), new ChangeColumnsValidator());

    public SchemaCompatibilityValidator(ValidationSchemasSource validationSchemasSource, CatalogService catalogService, SchemaSyncService schemaSyncService) {
        this.validationSchemasSource = validationSchemasSource;
        this.catalogService = catalogService;
        this.schemaSyncService = schemaSyncService;
    }

    public CompletableFuture<CompatValidationResult> validateCommit(UUID txId, Set<Integer> enlistedTableIds, HybridTimestamp commitTimestamp) {
        HybridTimestamp beginTimestamp = TransactionIds.beginTimestamp(txId);
        assert (commitTimestamp.compareTo(beginTimestamp) > 0);
        return this.schemaSyncService.waitForMetadataCompleteness(commitTimestamp).thenApply(ignored -> this.validateCommit(enlistedTableIds, commitTimestamp, beginTimestamp));
    }

    private CompatValidationResult validateCommit(Set<Integer> tableIds, HybridTimestamp commitTimestamp, HybridTimestamp beginTimestamp) {
        for (int tableId : tableIds) {
            CompatValidationResult validationResult = this.validateCommit(beginTimestamp, commitTimestamp, tableId);
            if (validationResult.isSuccessful()) continue;
            return validationResult;
        }
        return CompatValidationResult.success();
    }

    private CompatValidationResult validateCommit(HybridTimestamp beginTimestamp, HybridTimestamp commitTimestamp, int tableId) {
        CatalogTableDescriptor tableAtCommitTs = this.catalogService.activeCatalog(commitTimestamp.longValue()).table(tableId);
        if (tableAtCommitTs == null || tableAtCommitTs.isLockedForAccess()) {
            CatalogTableDescriptor tableAtTxStart = this.catalogService.activeCatalog(beginTimestamp.longValue()).table(tableId);
            assert (tableAtTxStart != null) : "No table " + tableId + " at ts " + beginTimestamp;
            return tableAtCommitTs == null ? CompatValidationResult.tableDropped(tableAtTxStart.name(), tableAtTxStart.schemaId()) : CompatValidationResult.tableLocked(tableAtTxStart.name(), tableAtTxStart.schemaId());
        }
        return this.validateForwardSchemaCompatibility(beginTimestamp, commitTimestamp, tableId);
    }

    private CompatValidationResult validateForwardSchemaCompatibility(HybridTimestamp beginTimestamp, HybridTimestamp commitTimestamp, int tableId) {
        List<FullTableSchema> tableSchemas = this.validationSchemasSource.tableSchemaVersionsBetween(tableId, beginTimestamp, commitTimestamp);
        assert (!tableSchemas.isEmpty());
        for (int i = 0; i < tableSchemas.size() - 1; ++i) {
            FullTableSchema oldSchema = tableSchemas.get(i);
            FullTableSchema newSchema = tableSchemas.get(i + 1);
            ValidationResult validationResult = this.validateForwardSchemaCompatibility(oldSchema, newSchema);
            if (validationResult.verdict != ValidatorVerdict.INCOMPATIBLE) continue;
            return CompatValidationResult.incompatibleChange(oldSchema.tableName(), oldSchema.schemaVersion(), newSchema.schemaVersion(), validationResult.details());
        }
        return CompatValidationResult.success();
    }

    private ValidationResult validateForwardSchemaCompatibility(FullTableSchema prevSchema, FullTableSchema nextSchema) {
        TableDefinitionDiff diff = this.diffCache.computeIfAbsent(new TableDefinitionDiffKey(prevSchema.tableId(), prevSchema.schemaVersion(), nextSchema.schemaVersion()), key -> nextSchema.diffFrom(prevSchema));
        boolean accepted = false;
        for (ForwardCompatibilityValidator validator : FORWARD_COMPATIBILITY_VALIDATORS) {
            ValidationResult validationResult = validator.compatible(diff);
            switch (validationResult.verdict) {
                case COMPATIBLE: {
                    accepted = true;
                    break;
                }
                case INCOMPATIBLE: {
                    return validationResult;
                }
            }
        }
        assert (accepted) : "Table schema changed from " + prevSchema.schemaVersion() + " to " + nextSchema.schemaVersion() + ", but no schema change validator voted for any change. Some schema validator is missing.";
        return ValidationResult.COMPATIBLE;
    }

    public CompletableFuture<CompatValidationResult> validateBackwards(int tupleSchemaVersion, int tableId, UUID txId) {
        HybridTimestamp beginTimestamp = TransactionIds.beginTimestamp(txId);
        return ((CompletableFuture)this.schemaSyncService.waitForMetadataCompleteness(beginTimestamp).thenCompose(ignored -> this.validationSchemasSource.waitForSchemaAvailability(tableId, tupleSchemaVersion))).thenApply(ignored -> this.validateBackwardSchemaCompatibility(tupleSchemaVersion, tableId, beginTimestamp));
    }

    private CompatValidationResult validateBackwardSchemaCompatibility(int tupleSchemaVersion, int tableId, HybridTimestamp beginTimestamp) {
        List<FullTableSchema> tableSchemas = this.validationSchemasSource.tableSchemaVersionsBetween(tableId, beginTimestamp, tupleSchemaVersion);
        if (tableSchemas.size() < 2) {
            return CompatValidationResult.success();
        }
        FullTableSchema oldSchema = tableSchemas.get(0);
        FullTableSchema newSchema = tableSchemas.get(1);
        return CompatValidationResult.incompatibleChange(oldSchema.tableName(), oldSchema.schemaVersion(), newSchema.schemaVersion(), null);
    }

    public void failIfSchemaChangedAfterTxStart(UUID txId, HybridTimestamp operationTimestamp, int tableId) {
        HybridTimestamp beginTs = TransactionIds.beginTimestamp(txId);
        CatalogTableDescriptor tableAtBeginTs = this.catalogService.activeCatalog(beginTs.longValue()).table(tableId);
        CatalogTableDescriptor tableAtOpTs = this.catalogService.activeCatalog(operationTimestamp.longValue()).table(tableId);
        assert (tableAtBeginTs != null) : "No table " + tableId + " at ts " + tableAtBeginTs;
        if (tableAtOpTs == null) {
            throw IncompatibleSchemaVersionException.tableDropped(tableAtBeginTs.name());
        }
        if (tableAtOpTs.latestSchemaVersion() != tableAtBeginTs.latestSchemaVersion()) {
            throw IncompatibleSchemaVersionException.schemaChanged(tableAtBeginTs.name(), tableAtBeginTs.latestSchemaVersion(), tableAtOpTs.latestSchemaVersion());
        }
    }

    public void failIfTableDoesNotExistAt(HybridTimestamp operationTimestamp, int tableId) {
        CatalogTableDescriptor tableAtOpTs = this.catalogService.activeCatalog(operationTimestamp.longValue()).table(tableId);
        if (tableAtOpTs == null) {
            throw IncompatibleSchemaVersionException.tableDropped(tableId);
        }
        if (tableAtOpTs.isLockedForAccess()) {
            throw IncompatibleSchemaVersionException.tableLocked(tableId);
        }
    }

    public void failIfRequestSchemaDiffersFromTxTs(HybridTimestamp txTs, int requestSchemaVersion, int tableId) {
        CatalogTableDescriptor table = this.catalogService.activeCatalog(txTs.longValue()).table(tableId);
        assert (table != null) : "No table " + tableId + " at " + txTs;
        if (table.latestSchemaVersion() != requestSchemaVersion) {
            throw new InternalSchemaVersionMismatchException();
        }
    }

    private static class ValidationResult {
        private static final ValidationResult COMPATIBLE = new ValidationResult(ValidatorVerdict.COMPATIBLE, null);
        private static final ValidationResult DONT_CARE = new ValidationResult(ValidatorVerdict.DONT_CARE, null);
        private final ValidatorVerdict verdict;
        private final String details;

        ValidationResult(ValidatorVerdict verdict, @Nullable String details) {
            this.verdict = verdict;
            this.details = details;
        }

        ValidatorVerdict verdict() {
            return this.verdict;
        }

        @Nullable
        String details() {
            return this.details;
        }
    }

    private static enum ValidatorVerdict {
        COMPATIBLE,
        INCOMPATIBLE,
        DONT_CARE;

    }

    private static interface ForwardCompatibilityValidator {
        public ValidationResult compatible(TableDefinitionDiff var1);
    }

    private static class RenameTableValidator
    implements ForwardCompatibilityValidator {
        private static final ValidationResult INCOMPATIBLE = new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Name of the table has been changed");

        private RenameTableValidator() {
        }

        @Override
        public ValidationResult compatible(TableDefinitionDiff diff) {
            return diff.nameDiffers() ? INCOMPATIBLE : ValidationResult.DONT_CARE;
        }
    }

    private static class AddColumnsValidator
    implements ForwardCompatibilityValidator {
        private AddColumnsValidator() {
        }

        @Override
        public ValidationResult compatible(TableDefinitionDiff diff) {
            if (diff.addedColumns().isEmpty()) {
                return ValidationResult.DONT_CARE;
            }
            for (CatalogTableColumnDescriptor column : diff.addedColumns()) {
                if (column.nullable() || column.defaultValue() != null) continue;
                return new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Not null column added without default value");
            }
            return ValidationResult.COMPATIBLE;
        }
    }

    private static class DropColumnsValidator
    implements ForwardCompatibilityValidator {
        private static final ValidationResult INCOMPATIBLE = new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Columns were dropped");

        private DropColumnsValidator() {
        }

        @Override
        public ValidationResult compatible(TableDefinitionDiff diff) {
            return diff.removedColumns().isEmpty() ? ValidationResult.DONT_CARE : INCOMPATIBLE;
        }
    }

    private static class ChangeColumnsValidator
    implements ForwardCompatibilityValidator {
        private static final List<ColumnChangeCompatibilityValidator> validators = List.of(new ChangeNullabilityValidator(), new ChangeDefaultValueValidator(), new ChangeColumnTypeValidator());

        private ChangeColumnsValidator() {
        }

        @Override
        public ValidationResult compatible(TableDefinitionDiff diff) {
            if (diff.changedColumns().isEmpty()) {
                return ValidationResult.DONT_CARE;
            }
            boolean accepted = false;
            for (ColumnDefinitionDiff columnDiff : diff.changedColumns()) {
                ValidationResult validationResult = ChangeColumnsValidator.compatible(columnDiff);
                switch (validationResult.verdict()) {
                    case COMPATIBLE: {
                        accepted = true;
                        break;
                    }
                    case INCOMPATIBLE: {
                        return validationResult;
                    }
                }
            }
            assert (accepted) : "Table schema changed from " + diff.oldSchemaVersion() + " to " + diff.newSchemaVersion() + ", but no column change validator voted for any change. Some schema validator is missing.";
            return ValidationResult.COMPATIBLE;
        }

        private static ValidationResult compatible(ColumnDefinitionDiff columnDiff) {
            boolean accepted = false;
            for (ColumnChangeCompatibilityValidator validator : validators) {
                ValidationResult validationResult = validator.compatible(columnDiff);
                switch (validationResult.verdict()) {
                    case COMPATIBLE: {
                        accepted = true;
                        break;
                    }
                    case INCOMPATIBLE: {
                        return validationResult;
                    }
                }
            }
            return accepted ? ValidationResult.COMPATIBLE : ValidationResult.DONT_CARE;
        }
    }

    private static class ChangeColumnTypeValidator
    implements ColumnChangeCompatibilityValidator {
        private ChangeColumnTypeValidator() {
        }

        @Override
        public ValidationResult compatible(ColumnDefinitionDiff diff) {
            if (!diff.typeChanged()) {
                return ValidationResult.DONT_CARE;
            }
            return diff.typeChangeIsSupported() ? ValidationResult.COMPATIBLE : new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Column type changed incompatibly");
        }
    }

    private static class ChangeNullabilityValidator
    implements ColumnChangeCompatibilityValidator {
        private ChangeNullabilityValidator() {
        }

        @Override
        public ValidationResult compatible(ColumnDefinitionDiff diff) {
            if (diff.notNullAdded()) {
                return new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Not null added");
            }
            if (diff.notNullDropped()) {
                return ValidationResult.COMPATIBLE;
            }
            assert (!diff.nullabilityChanged()) : diff;
            return ValidationResult.DONT_CARE;
        }
    }

    private static class ChangeDefaultValueValidator
    implements ColumnChangeCompatibilityValidator {
        private ChangeDefaultValueValidator() {
        }

        @Override
        public ValidationResult compatible(ColumnDefinitionDiff diff) {
            return diff.defaultChanged() ? new ValidationResult(ValidatorVerdict.INCOMPATIBLE, "Column default value changed") : ValidationResult.DONT_CARE;
        }
    }

    private static interface ColumnChangeCompatibilityValidator {
        public ValidationResult compatible(ColumnDefinitionDiff var1);
    }
}

