/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.sql.engine.exec.memory.structures.file;

import it.unimi.dsi.fastutil.longs.LongIterator;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.NoSuchElementException;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.ignite3.internal.fileio.FileIo;
import org.apache.ignite3.internal.fileio.FileIoFactory;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.schema.BinaryTuple;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.RowQueue;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.file.DataDirectory;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.file.ExternalCollectionUtils;
import org.apache.ignite3.internal.sql.engine.exec.memory.structures.file.ExternalFileStore;
import org.apache.ignite3.internal.util.IgniteUtils;
import org.apache.ignite3.sql.SqlException;
import org.gridgain.lang.GridgainErrorGroups;
import org.jetbrains.annotations.Nullable;

@NotThreadSafe
public class SortedExternalRowStore
implements AutoCloseable,
RowQueue<BinaryTuple> {
    private static final IgniteLogger LOG = Loggers.forClass(SortedExternalRowStore.class);
    private final OffsetStore indexStore;
    private final Comparator<ByteBuffer> tupleComparator;
    private boolean isClosed;
    private final ExternalFileStore externalStore;
    private int storedCount;
    private boolean sorted;

    public SortedExternalRowStore(Comparator<ByteBuffer> tupleComp, FileIoFactory fileIoFactory, DataDirectory workDir, int columnCount) {
        this.tupleComparator = tupleComp;
        this.externalStore = new ExternalFileStore(fileIoFactory, workDir, columnCount);
        this.indexStore = new OffsetStore(fileIoFactory, workDir);
    }

    @Override
    public void add(BinaryTuple row) {
        this.checkClosed();
        assert (!this.sorted);
        ++this.storedCount;
        long rowAddress = this.externalStore.write(row);
        this.indexStore.write(rowAddress);
    }

    private void sortIfNeeded() {
        this.checkClosed();
        this.sort0();
    }

    @Override
    public int size() {
        this.checkClosed();
        if (this.storedCount == 0) {
            return 0;
        }
        return this.externalStore.size();
    }

    @Override
    public void clear() {
        this.checkClosed();
        this.sorted = false;
        this.storedCount = 0;
        this.externalStore.reset();
        this.indexStore.reset();
    }

    @Override
    public void close() {
        if (this.isClosed) {
            return;
        }
        this.isClosed = true;
        IgniteUtils.closeQuiet(this.externalStore);
        IgniteUtils.closeQuiet(this.indexStore);
    }

    @Override
    public boolean isEmpty() {
        this.checkClosed();
        return this.size() == 0;
    }

    @Override
    @Nullable
    public BinaryTuple peek() {
        this.checkClosed();
        this.sortIfNeeded();
        Iterator<BinaryTuple> it = this.innerIterator();
        return it.hasNext() ? it.next() : null;
    }

    @Override
    @Nullable
    public BinaryTuple poll() {
        this.checkClosed();
        this.sortIfNeeded();
        Iterator<BinaryTuple> it = this.innerIterator();
        if (it.hasNext()) {
            BinaryTuple data = it.next();
            it.remove();
            return data;
        }
        return null;
    }

    @Override
    public BinaryTuple remove() {
        throw new UnsupportedOperationException("Unsupported operation: remove");
    }

    @Override
    public Iterator<BinaryTuple> iterator() {
        this.checkClosed();
        this.sortIfNeeded();
        return new Iterator<BinaryTuple>(){
            private final Iterator<BinaryTuple> it;
            {
                this.it = SortedExternalRowStore.this.innerIterator();
            }

            @Override
            public boolean hasNext() {
                SortedExternalRowStore.this.checkClosed();
                return this.it.hasNext();
            }

            @Override
            public BinaryTuple next() {
                SortedExternalRowStore.this.checkClosed();
                return this.it.next();
            }
        };
    }

    private Iterator<BinaryTuple> innerIterator() {
        if (this.storedCount == 0) {
            return Collections.emptyIterator();
        }
        return new Iterator<BinaryTuple>(){
            private final LongIterator sortedIt;
            long offset;
            {
                this.sortedIt = SortedExternalRowStore.this.indexStore.offsetsIterator();
                this.offset = -1L;
            }

            @Override
            public boolean hasNext() {
                return this.sortedIt.hasNext();
            }

            @Override
            public BinaryTuple next() {
                this.offset = this.sortedIt.next();
                BinaryTuple data = SortedExternalRowStore.this.externalStore.read(this.offset);
                assert (data != null);
                return data;
            }

            @Override
            public void remove() {
                if (this.offset == -1L) {
                    throw new IllegalStateException();
                }
                SortedExternalRowStore.this.externalStore.remove(this.offset);
                this.offset = -1L;
                this.sortedIt.remove();
            }
        };
    }

    private void checkClosed() {
        if (this.isClosed) {
            throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Sorted row store has been closed.");
        }
    }

    private void sort0() {
        int i;
        if (this.sorted || this.isEmpty()) {
            return;
        }
        int n = this.storedCount;
        for (i = (n >>> 1) - 1; i >= 0; --i) {
            this.heapify(i, n);
        }
        for (i = n - 1; i >= 0; --i) {
            this.indexStore.swapPositions(0, i);
            this.heapify(0, i);
        }
        this.sorted = true;
    }

    @Nullable
    private BinaryTuple readEntry(int pos) {
        long off = this.indexStore.getOffset(pos);
        return this.externalStore.read(off);
    }

    private <T> void heapify(int k, int n) {
        long refAddr = this.indexStore.getOffset(k);
        BinaryTuple refData = this.externalStore.read(refAddr);
        assert (refData != null);
        int half = n >>> 1;
        while (k < half) {
            int child = (k << 1) + 1;
            BinaryTuple c = this.readEntry(child);
            long offset = this.indexStore.getOffset(child);
            int right = child + 1;
            if (right < n && this.tupleComparator.compare(c.byteBuffer(), this.readEntry(right).byteBuffer()) < 0) {
                child = right;
                c = this.readEntry(child);
                offset = this.indexStore.getOffset(child);
            }
            if (this.tupleComparator.compare(refData.byteBuffer(), c.byteBuffer()) >= 0) break;
            this.indexStore.putOffset(k, offset);
            k = child;
        }
        this.indexStore.putOffset(k, refAddr);
    }

    @NotThreadSafe
    static class OffsetStore
    implements AutoCloseable {
        private final Path file;
        private final FileIo fileIo;
        private boolean isClosed;
        private long lastWrittenOffset;
        private final ByteBuffer buffer = ByteBuffer.allocate(8);
        private long headAddr;

        <RowT> OffsetStore(FileIoFactory fileIoFactory, DataDirectory workDir) {
            this.file = workDir.createFile("sortidx");
            try {
                this.fileIo = fileIoFactory.create(this.file, StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE);
            }
            catch (Exception e) {
                throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Failed to create spill file.", (Throwable)e);
            }
        }

        void write(long data) {
            this.checkClosed();
            try {
                this.buffer.rewind();
                this.buffer.putLong(data);
                this.buffer.flip();
                this.fileIo.position(this.lastWrittenOffset);
                this.fileIo.writeFully(this.buffer);
                this.lastWrittenOffset = this.fileIo.position();
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
        }

        private long getOffset(int pos) {
            this.buffer.rewind();
            try {
                this.fileIo.position((long)pos * 8L);
                this.fileIo.readFully(this.buffer);
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
            this.buffer.flip();
            return this.buffer.getLong();
        }

        private void putOffset(int pos, long addr) {
            this.buffer.rewind();
            try {
                this.fileIo.position((long)pos * 8L);
                this.buffer.putLong(addr);
                this.buffer.flip();
                this.fileIo.writeFully(this.buffer);
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
        }

        private void swapPositions(int pos1, int pos2) {
            long tmp = this.getOffset(pos1);
            this.putOffset(pos1, this.getOffset(pos2));
            this.putOffset(pos2, tmp);
        }

        LongIterator offsetsIterator() {
            return new LongIterator(){
                private long lastReadPosition;
                private long headAddrStored;
                private long offset;
                {
                    this.lastReadPosition = headAddr;
                    this.headAddrStored = headAddr;
                    this.offset = -1L;
                }

                public boolean hasNext() {
                    this.advance();
                    return this.offset >= 0L;
                }

                public long nextLong() {
                    this.advance();
                    if (this.offset < 0L) {
                        throw new NoSuchElementException();
                    }
                    long offset0 = this.offset;
                    this.offset = -1L;
                    return offset0;
                }

                public void remove() {
                    headAddr += 8L;
                }

                private void advance() {
                    this.checkClosed();
                    if (this.offset >= 0L && this.headAddrStored == headAddr) {
                        return;
                    }
                    this.headAddrStored = headAddr;
                    this.lastReadPosition = Math.max(this.lastReadPosition, headAddr);
                    try {
                        fileIo.position(this.lastReadPosition);
                        buffer.rewind();
                        int ret = fileIo.readFully(buffer);
                        this.lastReadPosition = fileIo.position();
                        if (ret == -1) {
                            this.offset = -1L;
                        } else {
                            buffer.flip();
                            this.offset = buffer.getLong();
                        }
                    }
                    catch (IOException e) {
                        this.close();
                        throw ExternalCollectionUtils.accessFailedException(e);
                    }
                }
            };
        }

        private void checkClosed() {
            if (this.isClosed) {
                throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Offsets store has been closed.");
            }
        }

        @Override
        public void close() {
            if (this.isClosed) {
                return;
            }
            this.isClosed = true;
            IgniteUtils.closeQuiet(this.fileIo);
            if (!this.file.toFile().delete()) {
                LOG.info("Failed to remove spill file " + this.file.getFileName(), new Object[0]);
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("Spill file removed " + this.file.getFileName(), new Object[0]);
            }
        }

        void reset() {
            this.checkClosed();
            this.lastWrittenOffset = 0L;
            this.headAddr = 0L;
            try {
                this.fileIo.clear();
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("External row store file cleaned: " + this.file.getFileName(), new Object[0]);
            }
        }
    }
}

