/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.index;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.ignite3.internal.catalog.Catalog;
import org.apache.ignite3.internal.catalog.CatalogManager;
import org.apache.ignite3.internal.catalog.CatalogService;
import org.apache.ignite3.internal.catalog.descriptors.CatalogIndexDescriptor;
import org.apache.ignite3.internal.catalog.descriptors.CatalogIndexStatus;
import org.apache.ignite3.internal.catalog.events.CatalogEvent;
import org.apache.ignite3.internal.catalog.events.MakeIndexAvailableEventParameters;
import org.apache.ignite3.internal.catalog.events.RemoveIndexEventParameters;
import org.apache.ignite3.internal.catalog.events.StartBuildingIndexEventParameters;
import org.apache.ignite3.internal.close.ManuallyCloseable;
import org.apache.ignite3.internal.failure.FailureContext;
import org.apache.ignite3.internal.failure.FailureProcessor;
import org.apache.ignite3.internal.index.IndexBuildCompletionListener;
import org.apache.ignite3.internal.index.IndexBuilder;
import org.apache.ignite3.internal.index.IndexManagementUtils;
import org.apache.ignite3.internal.lang.ByteArray;
import org.apache.ignite3.internal.lang.NodeStoppingException;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.metastorage.Entry;
import org.apache.ignite3.internal.metastorage.MetaStorageManager;
import org.apache.ignite3.internal.metastorage.WatchEvent;
import org.apache.ignite3.internal.metastorage.dsl.Condition;
import org.apache.ignite3.internal.metastorage.dsl.Conditions;
import org.apache.ignite3.internal.metastorage.dsl.Operations;
import org.apache.ignite3.internal.util.CollectionUtils;
import org.apache.ignite3.internal.util.CompletableFutures;
import org.apache.ignite3.internal.util.ExceptionUtils;
import org.apache.ignite3.internal.util.IgniteBusyLock;
import org.apache.ignite3.internal.util.IgniteSpinBusyLock;
import org.apache.ignite3.internal.util.IgniteUtils;

class IndexAvailabilityController
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(IndexAvailabilityController.class);
    private final CatalogManager catalogManager;
    private final MetaStorageManager metaStorageManager;
    private final FailureProcessor failureProcessor;
    private final IndexBuilder indexBuilder;
    private final IgniteSpinBusyLock busyLock = new IgniteSpinBusyLock();
    private final AtomicBoolean stopGuard = new AtomicBoolean();

    IndexAvailabilityController(CatalogManager catalogManager, MetaStorageManager metaStorageManager, FailureProcessor failureProcessor, IndexBuilder indexBuilder) {
        this.catalogManager = catalogManager;
        this.metaStorageManager = metaStorageManager;
        this.failureProcessor = failureProcessor;
        this.indexBuilder = indexBuilder;
    }

    @Override
    public void close() {
        if (!this.stopGuard.compareAndSet(false, true)) {
            return;
        }
        this.busyLock.block();
    }

    public void start(long recoveryRevision) {
        IgniteUtils.inBusyLock((IgniteBusyLock)this.busyLock, () -> {
            this.addListeners(this.catalogManager, this.metaStorageManager, this.indexBuilder);
            int catalogVersion = this.catalogManager.latestCatalogVersion();
            Catalog catalog = this.catalogManager.catalog(catalogVersion);
            List futures = catalog.indexes().stream().map(indexDescriptor -> {
                switch (indexDescriptor.status()) {
                    case BUILDING: {
                        return this.recoveryForBuildingIndexBusy((CatalogIndexDescriptor)indexDescriptor, recoveryRevision, catalog);
                    }
                    case AVAILABLE: {
                        return this.recoveryForAvailableIndexBusy((CatalogIndexDescriptor)indexDescriptor, recoveryRevision);
                    }
                }
                return CompletableFutures.nullCompletedFuture();
            }).filter(Predicate.not(CompletableFuture::isDone)).collect(Collectors.toList());
            CompletableFuture.allOf((CompletableFuture[])futures.toArray(CompletableFuture[]::new)).whenComplete((unused, throwable) -> {
                if (throwable != null && !(ExceptionUtils.unwrapCause(throwable) instanceof NodeStoppingException)) {
                    this.failureProcessor.process(new FailureContext((Throwable)throwable, "Error when trying to recover index availability"));
                } else if (!futures.isEmpty()) {
                    LOG.debug("Successful recovery of index availability", new Object[0]);
                }
            });
        });
    }

    private void addListeners(CatalogService catalogService, MetaStorageManager metaStorageManager, IndexBuilder indexBuilder) {
        catalogService.listen(CatalogEvent.INDEX_BUILDING, parameters -> this.onIndexBuilding((StartBuildingIndexEventParameters)parameters).thenApply(unused -> false));
        catalogService.listen(CatalogEvent.INDEX_REMOVED, parameters -> this.onIndexRemoved((RemoveIndexEventParameters)parameters).thenApply(unused -> false));
        catalogService.listen(CatalogEvent.INDEX_AVAILABLE, parameters -> this.onIndexAvailable((MakeIndexAvailableEventParameters)parameters).thenApply(unused -> false));
        metaStorageManager.registerPrefixWatch(ByteArray.fromString("indexBuild.partition."), event -> this.onUpdatePartitionBuildIndexKey(event).thenApply(unused -> null));
        indexBuilder.listen(new IndexBuildCompletionListener(){

            @Override
            public void onBuildCompletion(int indexId, int tableId, int partitionId) {
                IndexAvailabilityController.this.onIndexBuildCompletionForPartition(indexId, partitionId);
            }
        });
    }

    private CompletableFuture<?> onIndexBuilding(StartBuildingIndexEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            int indexId = parameters.indexId();
            Catalog catalog = this.catalogManager.catalog(parameters.catalogVersion());
            assert (catalog != null) : "Catalog is null for version: " + parameters.catalogVersion();
            int partitions = IndexManagementUtils.getPartitionCountFromCatalog(catalog, indexId);
            return IndexManagementUtils.putBuildIndexMetastoreKeysIfAbsent(this.metaStorageManager, indexId, partitions);
        });
    }

    private CompletableFuture<?> onIndexRemoved(RemoveIndexEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            int indexId = parameters.indexId();
            int previousCatalogVersion = parameters.catalogVersion() - 1;
            Catalog catalog = this.catalogManager.catalog(previousCatalogVersion);
            assert (catalog != null) : "Catalog is null for version: " + previousCatalogVersion;
            CatalogIndexDescriptor indexBeforeRemoval = catalog.index(indexId);
            if (indexBeforeRemoval.status() == CatalogIndexStatus.STOPPING) {
                return CompletableFutures.nullCompletedFuture();
            }
            int partitions = IndexManagementUtils.getPartitionCountFromCatalog(catalog, indexId);
            ByteArray inProgressBuildIndexKey = IndexManagementUtils.inProgressBuildIndexMetastoreKey(indexId);
            List removePartitionBuildIndexMetastoreKeyOperations = IntStream.range(0, partitions).mapToObj(partitionId -> IndexManagementUtils.partitionBuildIndexMetastoreKey(indexId, partitionId)).map(Operations::remove).collect(Collectors.toList());
            return this.metaStorageManager.invoke((Condition)Conditions.exists(inProgressBuildIndexKey), CollectionUtils.concat(List.of(Operations.remove(inProgressBuildIndexKey)), removePartitionBuildIndexMetastoreKeyOperations), List.of(Operations.noop()));
        });
    }

    private CompletableFuture<?> onIndexAvailable(MakeIndexAvailableEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            ByteArray inProgressBuildIndexMetastoreKey = IndexManagementUtils.inProgressBuildIndexMetastoreKey(parameters.indexId());
            return IndexManagementUtils.removeMetastoreKeyIfPresent(this.metaStorageManager, inProgressBuildIndexMetastoreKey);
        });
    }

    private CompletableFuture<?> onUpdatePartitionBuildIndexKey(WatchEvent event) {
        return IgniteUtils.inBusyLockAsync(this.busyLock, () -> {
            if (!event.single()) {
                return CompletableFutures.nullCompletedFuture();
            }
            Entry entry = event.entryEvent().newEntry();
            if (entry.value() != null) {
                return CompletableFutures.nullCompletedFuture();
            }
            String partitionBuildIndexKey = IndexManagementUtils.toPartitionBuildIndexMetastoreKeyString(entry.key());
            int indexId = IndexManagementUtils.extractIndexIdFromPartitionBuildIndexKey(partitionBuildIndexKey);
            long metastoreRevision = entry.revision();
            if (IndexManagementUtils.isAnyMetastoreKeyPresentLocally(this.metaStorageManager, IndexManagementUtils.partitionBuildIndexMetastoreKeyPrefix(indexId), metastoreRevision) || IndexManagementUtils.isMetastoreKeyAbsentLocally(this.metaStorageManager, IndexManagementUtils.inProgressBuildIndexMetastoreKey(indexId), metastoreRevision)) {
                return CompletableFutures.nullCompletedFuture();
            }
            IndexManagementUtils.makeIndexAvailableInCatalogWithoutFuture(this.catalogManager, indexId, this.failureProcessor);
            return CompletableFutures.nullCompletedFuture();
        });
    }

    private void onIndexBuildCompletionForPartition(int indexId, int partitionId) {
        IgniteUtils.inBusyLock((IgniteBusyLock)this.busyLock, () -> {
            ByteArray partitionBuildIndexKey = IndexManagementUtils.partitionBuildIndexMetastoreKey(indexId, partitionId);
            this.metaStorageManager.invoke((Condition)Conditions.exists(partitionBuildIndexKey), Operations.remove(partitionBuildIndexKey), Operations.noop()).whenComplete((operationResult, throwable) -> {
                if (throwable != null && !(ExceptionUtils.unwrapCause(throwable) instanceof NodeStoppingException)) {
                    String errorMessage = String.format("Error processing the operation to delete the partition index building key: [indexId=%s, partitionId=%s]", indexId, partitionId);
                    this.failureProcessor.process(new FailureContext((Throwable)throwable, errorMessage));
                }
            });
        });
    }

    private CompletableFuture<?> recoveryForAvailableIndexBusy(CatalogIndexDescriptor indexDescriptor, long recoveryRevision) {
        assert (indexDescriptor.status() == CatalogIndexStatus.AVAILABLE) : indexDescriptor.id();
        int indexId = indexDescriptor.id();
        ByteArray inProgressBuildIndexMetastoreKey = IndexManagementUtils.inProgressBuildIndexMetastoreKey(indexId);
        if (IndexManagementUtils.isMetastoreKeyAbsentLocally(this.metaStorageManager, inProgressBuildIndexMetastoreKey, recoveryRevision)) {
            return CompletableFutures.nullCompletedFuture();
        }
        return IndexManagementUtils.removeMetastoreKeyIfPresent(this.metaStorageManager, inProgressBuildIndexMetastoreKey);
    }

    private CompletableFuture<?> recoveryForBuildingIndexBusy(CatalogIndexDescriptor indexDescriptor, long recoveryRevision, Catalog catalog) {
        assert (indexDescriptor.status() == CatalogIndexStatus.BUILDING) : indexDescriptor.id();
        int indexId = indexDescriptor.id();
        if (IndexManagementUtils.isMetastoreKeyAbsentLocally(this.metaStorageManager, IndexManagementUtils.inProgressBuildIndexMetastoreKey(indexId), recoveryRevision)) {
            return IndexManagementUtils.putBuildIndexMetastoreKeysIfAbsent(this.metaStorageManager, indexId, IndexManagementUtils.getPartitionCountFromCatalog(catalog, indexId));
        }
        if (!IndexManagementUtils.isAnyMetastoreKeyPresentLocally(this.metaStorageManager, IndexManagementUtils.partitionBuildIndexMetastoreKeyPrefix(indexId), recoveryRevision)) {
            IndexManagementUtils.makeIndexAvailableInCatalogWithoutFuture(this.catalogManager, indexId, this.failureProcessor);
        }
        return CompletableFutures.nullCompletedFuture();
    }
}

