/*
 * Decompiled with CFR 0.152.
 */
package org.gridgain.internal.encryption.utils;

import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;
import java.util.Random;
import javax.crypto.Cipher;
import javax.crypto.spec.ChaCha20ParameterSpec;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import org.apache.ignite3.internal.lang.IgniteInternalException;
import org.apache.ignite3.lang.ErrorGroups;
import org.apache.ignite3.lang.IgniteException;
import org.gridgain.internal.encryption.DataEncryptionKeySerializer;
import org.gridgain.internal.encryption.provider.DataEncryptionKey;
import org.jetbrains.annotations.Nullable;

public final class EncryptionUtils {
    private static final int GCM_TAG_LENGTH_BITS = 128;
    private static final int CHACHA20_MIN_BUFFER_SIZE = 42;
    private static final int CHACHA20_NONCE_SIZE = 12;

    private EncryptionUtils() {
    }

    public static void encrypt(Cipher cipher, Random rnd, ByteBuffer srcBuf, ByteBuffer res, Key key) {
        try {
            int blockSize = cipher.getBlockSize();
            ByteBuffer inputBuffer = srcBuf;
            if (EncryptionUtils.isNoPadding(cipher)) {
                inputBuffer = EncryptionUtils.padToBlockSize(srcBuf, blockSize);
            }
            if (EncryptionUtils.isEcbMode(cipher)) {
                cipher.init(1, key);
                byte[] dummyIv = new byte[blockSize];
                res.put(dummyIv);
                cipher.doFinal(inputBuffer, res);
            } else {
                byte[] iv;
                if (EncryptionUtils.isChaCha20(cipher)) {
                    iv = EncryptionUtils.initVector(rnd, 12);
                    byte[] storedIv = new byte[16];
                    System.arraycopy(iv, 0, storedIv, 0, iv.length);
                    iv = storedIv;
                } else {
                    iv = EncryptionUtils.initVector(rnd, blockSize);
                }
                AlgorithmParameterSpec paramSpec = EncryptionUtils.createParameterSpec(cipher, iv);
                cipher.init(1, key, paramSpec);
                res.put(iv);
                cipher.doFinal(inputBuffer, res);
            }
        }
        catch (GeneralSecurityException e) {
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] encryptKey(Random rnd, DataEncryptionKey dataKey, Key key, String cipherAlgorithm) {
        byte[] byArray;
        byte[] plainDataKeyBytes = null;
        try {
            Cipher cipher = EncryptionUtils.getCipher(cipherAlgorithm);
            plainDataKeyBytes = DataEncryptionKeySerializer.serialize(dataKey);
            int bufferSize = EncryptionUtils.encryptedSize(cipher, plainDataKeyBytes.length);
            if (EncryptionUtils.isChaCha20(cipher) && bufferSize < 42) {
                bufferSize = 42;
            }
            ByteBuffer res = ByteBuffer.allocate(bufferSize);
            EncryptionUtils.encrypt(cipher, rnd, ByteBuffer.wrap(plainDataKeyBytes), res, key);
            byArray = res.array();
        }
        catch (Throwable throwable) {
            EncryptionUtils.clear(plainDataKeyBytes);
            throw throwable;
        }
        EncryptionUtils.clear(plainDataKeyBytes);
        return byArray;
    }

    public static DataEncryptionKey decryptKey(byte[] encDataKey, Key key, String cipherAlgorithm) {
        byte[] plainDataKeyBytes = null;
        try {
            plainDataKeyBytes = new byte[encDataKey.length];
            Cipher cipher = EncryptionUtils.getCipher(cipherAlgorithm);
            EncryptionUtils.decrypt(cipher, ByteBuffer.wrap(encDataKey), ByteBuffer.wrap(plainDataKeyBytes), key);
            try {
                DataEncryptionKey dataEncryptionKey = DataEncryptionKeySerializer.deserialize(plainDataKeyBytes);
                return dataEncryptionKey;
            }
            catch (IgniteInternalException e) {
                throw new IgniteException(ErrorGroups.Common.INTERNAL_ERR, "Failed to deserialize data encryption key. Such issues can arise if a bad key is used during decryption.", (Throwable)e);
            }
        }
        finally {
            EncryptionUtils.clear(plainDataKeyBytes);
        }
    }

    public static void decrypt(Cipher cipher, ByteBuffer srcBuf, ByteBuffer destBuf, Key key) {
        try {
            int ivSize = cipher.getBlockSize();
            if (EncryptionUtils.isChaCha20(cipher)) {
                ivSize = 16;
            }
            byte[] iv = new byte[ivSize];
            srcBuf.get(iv);
            if (EncryptionUtils.isEcbMode(cipher)) {
                cipher.init(2, key);
            } else {
                AlgorithmParameterSpec paramSpec = EncryptionUtils.createParameterSpec(cipher, iv);
                cipher.init(2, key, paramSpec);
            }
            cipher.doFinal(srcBuf, destBuf);
        }
        catch (GeneralSecurityException e) {
            throw new IgniteException(ErrorGroups.Common.INTERNAL_ERR, (Throwable)e);
        }
    }

    public static void clear(byte @Nullable [] array) {
        if (array != null) {
            Arrays.fill(array, (byte)0);
        }
    }

    public static void clear(char @Nullable [] array) {
        if (array != null) {
            Arrays.fill(array, ' ');
        }
    }

    public static int encryptedSize(Cipher cipher, int plainDataSize) {
        int blockSize = cipher.getBlockSize();
        if (blockSize == 0) {
            if (EncryptionUtils.isChaCha20(cipher)) {
                return plainDataSize + 42;
            }
            if (EncryptionUtils.isGcmMode(cipher)) {
                return plainDataSize + 16 + 16;
            }
            return plainDataSize + 16;
        }
        int encryptedSize = plainDataSize + blockSize;
        if (EncryptionUtils.isGcmMode(cipher)) {
            encryptedSize += 16;
        }
        if (EncryptionUtils.isNoPadding(cipher)) {
            int dataSize = plainDataSize;
            int paddingNeeded = (blockSize - dataSize % blockSize) % blockSize;
            return encryptedSize + paddingNeeded;
        }
        return EncryptionUtils.roundToBlockSize(cipher, encryptedSize);
    }

    public static int roundToBlockSize(Cipher cipher, int value) {
        int blockSize = cipher.getBlockSize();
        if (blockSize == 0) {
            return value;
        }
        return (value / blockSize + 1) * blockSize;
    }

    private static AlgorithmParameterSpec createParameterSpec(Cipher cipher, byte[] iv) {
        String algorithm = cipher.getAlgorithm();
        if (EncryptionUtils.isGcmMode(cipher)) {
            return new GCMParameterSpec(128, iv);
        }
        if (algorithm.contains("ChaCha20")) {
            byte[] nonce = iv.length > 12 ? Arrays.copyOf(iv, 12) : iv;
            return new ChaCha20ParameterSpec(nonce, 1);
        }
        return new IvParameterSpec(iv);
    }

    private static boolean isGcmMode(Cipher cipher) {
        return cipher.getAlgorithm().contains("GCM");
    }

    private static boolean isEcbMode(Cipher cipher) {
        return cipher.getAlgorithm().contains("ECB");
    }

    private static boolean isNoPadding(Cipher cipher) {
        return cipher.getAlgorithm().endsWith("NoPadding");
    }

    private static boolean isChaCha20(Cipher cipher) {
        return cipher.getAlgorithm().contains("ChaCha20");
    }

    private static ByteBuffer padToBlockSize(ByteBuffer srcBuf, int blockSize) {
        int remaining = srcBuf.remaining();
        int paddingNeeded = (blockSize - remaining % blockSize) % blockSize;
        if (paddingNeeded == 0) {
            return srcBuf;
        }
        ByteBuffer paddedBuf = ByteBuffer.allocate(remaining + paddingNeeded);
        byte[] originalData = new byte[remaining];
        srcBuf.get(originalData);
        paddedBuf.put(originalData);
        for (int i = 0; i < paddingNeeded; ++i) {
            paddedBuf.put((byte)0);
        }
        paddedBuf.flip();
        return paddedBuf;
    }

    private static byte[] initVector(Random rnd, int size) {
        int ivSize = size > 0 ? size : 16;
        byte[] iv = new byte[ivSize];
        rnd.nextBytes(iv);
        return iv;
    }

    private static Cipher getCipher(String algorithm) {
        try {
            return Cipher.getInstance(algorithm);
        }
        catch (GeneralSecurityException e) {
            throw new IgniteException(ErrorGroups.Common.INTERNAL_ERR, (Throwable)e);
        }
    }
}

