/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.compute.state;

import java.time.Instant;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import org.apache.ignite3.compute.JobState;
import org.apache.ignite3.compute.JobStatus;
import org.apache.ignite3.internal.compute.Cleaner;
import org.apache.ignite3.internal.compute.JobStateImpl;
import org.apache.ignite3.internal.compute.configuration.ComputeConfiguration;
import org.apache.ignite3.internal.compute.state.ComputeStateMachine;
import org.apache.ignite3.internal.compute.state.IllegalJobStatusTransition;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;

public class InMemoryComputeStateMachine
implements ComputeStateMachine {
    private static final IgniteLogger LOG = Loggers.forClass(InMemoryComputeStateMachine.class);
    private final ComputeConfiguration configuration;
    private final String nodeName;
    private final Cleaner<JobState> cleaner = new Cleaner();
    private final Map<UUID, JobState> states = new ConcurrentHashMap<UUID, JobState>();

    public InMemoryComputeStateMachine(ComputeConfiguration configuration, String nodeName) {
        this.configuration = configuration;
        this.nodeName = nodeName;
    }

    @Override
    public void start() {
        long ttlMillis = (Long)this.configuration.statesLifetimeMillis().value();
        this.cleaner.start(this.states::remove, ttlMillis, this.nodeName);
    }

    @Override
    public void stop() {
        this.cleaner.stop();
    }

    @Override
    public JobState currentState(UUID jobId) {
        return this.states.get(jobId);
    }

    @Override
    public UUID initJob() {
        JobStateImpl state;
        UUID uuid = UUID.randomUUID();
        if (this.states.putIfAbsent(uuid, state = JobStateImpl.builder().id(uuid).status(JobStatus.QUEUED).createTime(Instant.now()).build()) != null) {
            LOG.info("UUID collision detected! UUID: {}", uuid);
            return this.initJob();
        }
        return uuid;
    }

    @Override
    public void executeJob(UUID jobId) {
        this.changeJobStatus(jobId, JobStatus.EXECUTING);
    }

    @Override
    public void failJob(UUID jobId) {
        this.changeJobStatus(jobId, JobStatus.FAILED);
        this.cleaner.scheduleRemove(jobId);
    }

    @Override
    public void queueJob(UUID jobId) {
        this.changeJobStatus(jobId, JobStatus.QUEUED);
    }

    @Override
    public void completeJob(UUID jobId) {
        this.changeJobStatus(jobId, JobStatus.COMPLETED);
        this.cleaner.scheduleRemove(jobId);
    }

    @Override
    public JobState cancelingJob(UUID jobId) {
        return this.changeJobStatus(jobId, (JobStatus currentStatus) -> {
            if (currentStatus == JobStatus.QUEUED) {
                this.cleaner.scheduleRemove(jobId);
                return JobStatus.CANCELED;
            }
            if (currentStatus == JobStatus.EXECUTING) {
                return JobStatus.CANCELING;
            }
            throw new IllegalJobStatusTransition(jobId, (JobStatus)((Object)currentStatus), JobStatus.CANCELING);
        });
    }

    @Override
    public void cancelJob(UUID jobId) {
        this.changeJobStatus(jobId, JobStatus.CANCELED);
        this.cleaner.scheduleRemove(jobId);
    }

    private void changeJobStatus(UUID jobId, JobStatus newStatus) {
        this.changeJobStatus(jobId, (JobStatus ignored) -> newStatus);
    }

    private JobState changeJobStatus(UUID jobId, Function<JobStatus, JobStatus> newStatusFunction) {
        return this.changeStatus(jobId, currentState -> {
            JobStatus currentStatus = currentState.status();
            JobStatus newStatus = (JobStatus)((Object)((Object)newStatusFunction.apply(currentStatus)));
            InMemoryComputeStateMachine.validateStatusTransition(jobId, currentStatus, newStatus);
            JobStateImpl.Builder builder = JobStateImpl.toBuilder(currentState).status(newStatus);
            if (newStatus == JobStatus.EXECUTING) {
                builder.startTime(Instant.now());
            } else if (InMemoryComputeStateMachine.isFinal(newStatus)) {
                builder.finishTime(Instant.now());
            }
            return builder.build();
        });
    }

    private JobState changeStatus(UUID jobId, Function<JobState, JobState> newStateFunction) {
        JobState state = this.states.computeIfPresent(jobId, (k, v) -> (JobState)newStateFunction.apply((JobState)v));
        if (state == null) {
            throw new IllegalJobStatusTransition(jobId);
        }
        return state;
    }

    private static boolean isFinal(JobStatus status) {
        return status == JobStatus.FAILED || status == JobStatus.COMPLETED || status == JobStatus.CANCELED;
    }

    private static void validateStatusTransition(UUID jobId, JobStatus current, JobStatus target) {
        if (!InMemoryComputeStateMachine.isValidStatusTransition(current, target)) {
            throw new IllegalJobStatusTransition(jobId, current, target);
        }
    }

    private static boolean isValidStatusTransition(JobStatus from, JobStatus toStatus) {
        switch (from) {
            case QUEUED: {
                return toStatus == JobStatus.EXECUTING || toStatus == JobStatus.CANCELING || toStatus == JobStatus.CANCELED;
            }
            case EXECUTING: {
                return toStatus == JobStatus.FAILED || toStatus == JobStatus.COMPLETED || toStatus == JobStatus.CANCELING || toStatus == JobStatus.CANCELED || toStatus == JobStatus.QUEUED;
            }
            case CANCELING: {
                return toStatus == JobStatus.CANCELED || toStatus == JobStatus.FAILED || toStatus == JobStatus.COMPLETED;
            }
            case FAILED: 
            case COMPLETED: 
            case CANCELED: {
                return false;
            }
        }
        throw new IllegalStateException("Unknown job status: " + String.valueOf((Object)from));
    }
}

