/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.spi.deployment.uri;

import java.io.File;
import java.io.FilenameFilter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import org.apache.ignite.IgniteLogger;
import org.apache.ignite.spi.IgniteSpiException;

/**
 * Factory class to create class loader that loads classes and resources from
 * the deployment archive or "unpacked" deployment archive (package directory).
 * <p>
 * Class loader scans deployment archive or directory first and than if
 * class/resource was not found scans all JAR files.
 * It is assumed that all libraries are in the {@link #DFLT_LIBS_DIR_PATH}
 * directory.
 */
class GridUriDeploymentClassLoaderFactory {
    /** Libraries directory default value (value is {@code lib}). */
    public static final String DFLT_LIBS_DIR_PATH = "lib";

    /**
     * @param parent Parent class loader.
     * @param file Deployment archive or directory with an unpacked deployment archive.
     * @param log Logger.
     * @return Class Loader.
     * @throws org.apache.ignite.spi.IgniteSpiException In case of any error.
     */
    public static ClassLoader create(ClassLoader parent, File file, IgniteLogger log) throws IgniteSpiException {
        assert parent != null;
        assert file != null;
        assert log != null;

        assert file.isDirectory();

        List<URL> urls = new ArrayList<>();

        try {
            String url = file.toURI().toURL().toString();

            URL mainUrl = url.length() > 0 && url.charAt(url.length() - 1) == '/' ?
                file.toURI().toURL() : new URL(url + '/');

            urls.add(mainUrl);

            File libDir = new File(file, DFLT_LIBS_DIR_PATH);

            if (libDir.exists()) {
                File[] files = libDir.listFiles(new FilenameFilter() {
                    @Override public boolean accept(File dir, String name) {
                        return name.endsWith(".jar");
                    }
                });

                if (files.length > 0) {
                    for (File jarFile : files) {
                        urls.add(jarFile.toURI().toURL());
                    }
                }
            }

            return new GridUriDeploymentClassLoader(urls.toArray(new URL[urls.size()]), parent);
        }
        catch (MalformedURLException e) {
            throw new IgniteSpiException("Failed to create class loader for a package: " + file, e);
        }
    }

    /**
     * Ensure singleton.
     */
    private GridUriDeploymentClassLoaderFactory() {
        // No-op.
    }
}