/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.sql.engine.exec.memory.structures.file;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Comparator;
import java.util.Iterator;
import java.util.function.ToIntFunction;
import org.apache.ignite.internal.binarytuple.BinaryTupleParser;
import org.apache.ignite.internal.fileio.FileIo;
import org.apache.ignite.internal.fileio.FileIoFactory;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.schema.BinaryTuple;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.DataDirectory;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.ExternalCollectionUtils;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.ExternalFileStore;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.sql.SqlException;
import org.gridgain.lang.GridgainErrorGroups;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

public class ExternalHashSet
implements AutoCloseable {
    private static final IgniteLogger log = Loggers.forClass(ExternalHashSet.class);
    static final int ENTRY_SIZE = 12;
    private static final double LOAD_FACTOR = 0.5;
    private static final double SHRINK_FACTOR = 0.45;
    private static final long MAX_CAPACITY = 0x400000000000000L;
    private static final long TOMBSTONE_FLAG = Long.MIN_VALUE;
    static final int DEFAULT_CAPACITY = 16;
    private final FileIoFactory fileIoFactory;
    private final DataDirectory workDir;
    private final ByteBuffer reusableBuff = ByteBuffer.allocate(12);
    private final Comparator<BinaryTuple> keyComparator = Comparator.comparing(BinaryTupleParser::byteBuffer);
    private final ToIntFunction<BinaryTuple> hashFunction = k -> k.byteBuffer().hashCode();
    private final ExternalFileStore rowDataStore;
    private FileIo fileIo;
    private Path file;
    private long capacity;
    private int usedSlots;
    private int removedEntries;
    private boolean isClosed = false;

    ExternalHashSet(FileIoFactory fileIoFactory, DataDirectory workDir, int capacity, int columnsCount) {
        this.workDir = workDir;
        this.fileIoFactory = fileIoFactory;
        this.rowDataStore = new ExternalFileStore(fileIoFactory, workDir, columnsCount);
        this.initNewIndexFile(capacity);
    }

    private int getHashCode(BinaryTuple row) {
        return this.hashFunction.applyAsInt(row);
    }

    public synchronized boolean contains(BinaryTuple key) {
        this.checkCancelled();
        Entry entry = this.findEntry(key, this.getHashCode(key));
        return entry != null;
    }

    @Nullable
    public synchronized BinaryTuple get(BinaryTuple key) {
        this.checkCancelled();
        Entry entry = this.findEntry(key, this.getHashCode(key));
        return entry == null ? null : this.rowDataStore.read(entry.rowAddress);
    }

    public synchronized boolean putIfAbsent(BinaryTuple row) {
        this.checkCancelled();
        int hashCode = this.getHashCode(row);
        Entry entry = this.findEntry(row, hashCode);
        if (entry == null) {
            this.ensureCapacity();
            long address = this.rowDataStore.write(row);
            this.putEntryToFreeSlot(hashCode, address);
            return true;
        }
        return false;
    }

    public synchronized Iterator<BinaryTuple> iterator() {
        this.checkCancelled();
        return this.rowDataStore.iterator();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reset() {
        Path idxFile;
        ExternalHashSet externalHashSet = this;
        synchronized (externalHashSet) {
            this.checkCancelled();
            idxFile = this.file;
            this.capacity = 16L;
            this.usedSlots = 0;
            this.removedEntries = 0;
            this.reusableBuff.clear();
            try {
                this.fileIo.clear();
                this.fileIo.write(this.reusableBuff, this.capacity * 12L);
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
            this.rowDataStore.reset();
        }
        if (log.isDebugEnabled()) {
            log.debug("External row store file cleaned: " + idxFile.getFileName(), new Object[0]);
        }
    }

    public synchronized int size() {
        this.checkCancelled();
        return this.rowDataStore.size();
    }

    @TestOnly
    public synchronized int capacity() {
        this.checkCancelled();
        return (int)this.capacity;
    }

    @Nullable
    public synchronized BinaryTuple remove(BinaryTuple key) {
        this.checkCancelled();
        int hashCode = this.getHashCode(key);
        Entry entry = this.findEntry(key, hashCode);
        if (entry == null) {
            return null;
        }
        this.writeEntryToIndexFile(entry.slot(), hashCode, entry.rowAddress() | Long.MIN_VALUE);
        BinaryTuple result = this.rowDataStore.read(entry.rowAddress);
        this.rowDataStore.remove(entry.rowAddress);
        ++this.removedEntries;
        this.shrinkToFit();
        return result;
    }

    private void putEntryToFreeSlot(int hashCode, long rowAddr) {
        long slot = this.findFreeSlotForInsert(hashCode);
        this.writeEntryToIndexFile(slot, hashCode, rowAddr);
        ++this.usedSlots;
    }

    private long findFreeSlotForInsert(int hashCode) {
        long slot;
        long startSlot = slot = this.slot(hashCode);
        Entry entry = this.readEntryFromIndexFile(slot);
        while (!entry.isTombstone() && !entry.isEmpty()) {
            slot = (slot + 1L) % this.capacity;
            assert (slot != startSlot);
            entry = this.readEntryFromIndexFile(slot);
        }
        return slot;
    }

    private long slot(long hashCode) {
        long hc64 = hashCode << 48 ^ hashCode << 32 ^ hashCode << 16 ^ hashCode;
        return hc64 & this.capacity - 1L;
    }

    @Nullable
    private Entry findEntry(BinaryTuple key, int hashCode) {
        long slot;
        long initialSlot = slot = this.slot(hashCode);
        Entry entry = this.readEntryFromIndexFile(slot);
        while (!entry.isEmpty()) {
            if (!entry.isTombstone() && hashCode == entry.hash()) {
                BinaryTuple foundRow = this.rowDataStore.read(entry.rowAddress());
                assert (foundRow != null);
                if (this.keyComparator.compare(key, foundRow) == 0) break;
            }
            if ((slot = (slot + 1L) % this.capacity) == initialSlot) {
                return null;
            }
            entry = this.readEntryFromIndexFile(slot);
        }
        return entry.isEmpty() || entry.isTombstone() ? null : entry;
    }

    private Entry readEntryFromIndexFile(long slot) {
        return this.readEntryFromIndexFile(slot, this.fileIo);
    }

    private Entry readEntryFromIndexFile(long slot, FileIo fileCh) {
        try {
            if (slot != -1L) {
                this.gotoSlot(slot);
            } else {
                slot = fileCh.position() / 12L;
            }
            this.reusableBuff.clear();
            fileCh.readFully(this.reusableBuff);
            this.reusableBuff.flip();
            int hashCode = this.reusableBuff.getInt();
            long addr = this.reusableBuff.getLong() - 1L;
            return new Entry(hashCode, addr, slot);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void writeEntryToIndexFile(long slot, int hashCode, long addr) {
        try {
            this.gotoSlot(slot);
            this.reusableBuff.clear();
            this.reusableBuff.putInt(hashCode);
            this.reusableBuff.putLong(addr + 1L);
            this.reusableBuff.flip();
            this.fileIo.writeFully(this.reusableBuff);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void gotoSlot(long slot) {
        try {
            this.fileIo.position(slot * 12L);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureCapacity() {
        if ((double)this.usedSlots <= 0.5 * (double)this.capacity) {
            return;
        }
        FileIo oldFileIo = this.fileIo;
        File oldIdxFile = this.file.toFile();
        long oldSize = this.capacity;
        try {
            this.initNewIndexFile(oldSize * 2L);
            this.copyDataFromOldFile(oldFileIo, oldSize);
        }
        finally {
            IgniteUtils.closeQuiet((AutoCloseable)oldFileIo);
            this.deleteUnusedFile(oldIdxFile);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void shrinkToFit() {
        if ((double)this.removedEntries <= 0.45 * (double)this.capacity) {
            return;
        }
        FileIo oldFileIo = this.fileIo;
        File oldIdxFile = this.file.toFile();
        long oldSize = this.capacity;
        try {
            this.initNewIndexFile(oldSize / 2L);
            this.copyDataFromOldFile(oldFileIo, oldSize);
        }
        finally {
            IgniteUtils.closeQuiet((AutoCloseable)oldFileIo);
            this.deleteUnusedFile(oldIdxFile);
        }
    }

    private void copyDataFromOldFile(FileIo oldFile, long oldSize) {
        try {
            this.usedSlots = 0;
            this.removedEntries = 0;
            oldFile.position(0L);
            for (long i = 0L; i < oldSize; ++i) {
                Entry e = this.readEntryFromIndexFile(-1L, oldFile);
                if (e.isTombstone() || e.isEmpty()) continue;
                this.putEntryToFreeSlot(e.hash(), e.rowAddress());
            }
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)oldFile);
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initNewIndexFile(long cap) {
        try {
            assert (cap > 0L && (cap & cap - 1L) == 0L) : "cap=" + cap;
            if (cap > 0x400000000000000L) {
                throw new IllegalArgumentException("Maximum capacity is exceeded [curCapacity=" + cap + ", maxCapacity=288230376151711744]");
            }
            Path newFile = this.workDir.createFile("hashIdx");
            ExternalHashSet externalHashSet = this;
            synchronized (externalHashSet) {
                this.capacity = cap;
                this.file = newFile;
                this.fileIo = this.fileIoFactory.create(this.file, new OpenOption[]{StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE});
                this.reusableBuff.clear();
                this.fileIo.write(this.reusableBuff, cap * 12L);
            }
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void checkCancelled() {
        if (this.isClosed) {
            throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Set row store has been closed.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        ExternalHashSet externalHashSet = this;
        synchronized (externalHashSet) {
            if (this.isClosed) {
                return;
            }
            this.isClosed = true;
        }
        IgniteUtils.closeQuiet((AutoCloseable)this.fileIo);
        this.rowDataStore.close();
        this.deleteUnusedFile(this.file.toFile());
    }

    private void deleteUnusedFile(File file) {
        if (!file.delete()) {
            log.info("Failed to remove spill file " + file.getName(), new Object[0]);
        } else if (log.isDebugEnabled()) {
            log.debug("Spill file removed " + file.getName(), new Object[0]);
        }
    }

    private static class Entry {
        private final int hashCode;
        private final long rowAddress;
        private final long slot;

        Entry(int hashCode, long rowAddress, long slot) {
            this.hashCode = hashCode;
            this.rowAddress = rowAddress;
            this.slot = slot;
        }

        int hash() {
            return this.hashCode;
        }

        long rowAddress() {
            return this.rowAddress & Long.MAX_VALUE;
        }

        long slot() {
            return this.slot;
        }

        boolean isTombstone() {
            return (this.rowAddress & Long.MIN_VALUE) != 0L;
        }

        boolean isEmpty() {
            return this.rowAddress == -1L;
        }
    }
}

