/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.sql.engine.exec.memory.structures.file;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import org.apache.ignite.internal.binarytuple.BinaryTupleParser;
import org.apache.ignite.internal.fileio.FileIo;
import org.apache.ignite.internal.fileio.FileIoFactory;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.schema.BinaryTuple;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.DataDirectory;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.ExternalCollectionUtils;
import org.apache.ignite.internal.sql.engine.exec.memory.structures.file.ExternalFileStore;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.sql.SqlException;
import org.gridgain.lang.GridgainErrorGroups;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

public class ExternalHashTable
implements AutoCloseable {
    private static final IgniteLogger log = Loggers.forClass(ExternalHashTable.class);
    private static final double LOAD_FACTOR = 0.5;
    private static final long MAX_CAPACITY = 0x400000000000000L;
    private static final long TOMBSTONE_FLAG = Long.MIN_VALUE;
    private static final int DEFAULT_CAPACITY = 16;
    static final int ENTRY_SIZE = 12;
    static final int ROW_HEADER_SIZE = 12;
    private final FileIoFactory fileIoFactory;
    private final DataDirectory workDir;
    private final ByteBuffer reusableBuff = ByteBuffer.allocate(12);
    private final Comparator<BinaryTuple> keyComparator = Comparator.comparing(BinaryTupleParser::byteBuffer);
    private final ToIntFunction<BinaryTuple> hashFunction = k -> k.byteBuffer().hashCode();
    private final ExternalFileStore rowDataStore;
    private final ExternalRowReader tableRowReader = new ExternalRowReader();
    private ByteBuffer rowBuffer = ExternalHashTable.allocateNewBuffer(128);
    private FileIo fileIo;
    private Path file;
    private long capacity = 16L;
    private int usedSlots;
    private boolean isClosed = false;

    public ExternalHashTable(DataDirectory workDir, FileIoFactory fileIoFactory, int capacity) {
        this.workDir = workDir;
        this.fileIoFactory = fileIoFactory;
        this.rowDataStore = new ExternalFileStore(fileIoFactory, workDir, 0);
        this.initNewIndexFile(capacity);
    }

    private int getHashCode(BinaryTuple row) {
        return this.hashFunction.applyAsInt(row);
    }

    public synchronized boolean contains(BinaryTuple key) {
        this.checkCancelled();
        Entry entry = this.findEntry(key, this.getHashCode(key));
        return entry != null;
    }

    @Nullable
    public synchronized ByteBuffer get(BinaryTuple key) {
        this.checkCancelled();
        int hashCode = this.getHashCode(key);
        Entry entry = this.findEntryAndLoadRow(key, hashCode, true);
        if (entry != null) {
            return ExternalHashTable.copyValueFromRowBuffer(this.rowBuffer, this.tableRowReader.rowSize);
        }
        return null;
    }

    public synchronized void put(BinaryTuple key, ByteBuffer valueBuff) {
        this.checkCancelled();
        int hashCode = this.getHashCode(key);
        Entry entry = this.findEntry(key, hashCode);
        ByteBuffer keyBuf = key.byteBuffer();
        if (entry != null) {
            this.modifyOrReplaceEntry(entry, keyBuf, key.elementCount(), valueBuff);
        } else {
            this.insertEntry(hashCode, keyBuf, key.elementCount(), valueBuff);
        }
    }

    @Nullable
    public synchronized ByteBuffer remove(BinaryTuple key) {
        this.checkCancelled();
        int hashCode = this.getHashCode(key);
        Entry entry = this.findEntryAndLoadRow(key, hashCode, true);
        if (entry == null) {
            return null;
        }
        this.removeEntry(entry, hashCode);
        return ExternalHashTable.copyValueFromRowBuffer(this.rowBuffer, this.tableRowReader.rowSize);
    }

    public synchronized ByteBuffer computeIfAbsent(BinaryTuple key, Function<BinaryTuple, ByteBuffer> function) {
        this.checkCancelled();
        int hashCode = this.getHashCode(key);
        Entry entry = this.findEntryAndLoadRow(key, hashCode, true);
        if (entry != null) {
            return ExternalHashTable.copyValueFromRowBuffer(this.rowBuffer, this.tableRowReader.rowSize);
        }
        ByteBuffer keyBuf = key.byteBuffer();
        ByteBuffer valueBuff = function.apply(key);
        this.insertEntry(hashCode, keyBuf, key.elementCount(), valueBuff);
        valueBuff.flip();
        return valueBuff;
    }

    public synchronized Iterator<Map.Entry<BinaryTuple, ByteBuffer>> entryIterator() {
        this.checkCancelled();
        ExternalFileStore.RowReader rowReader = (rowSize, reader) -> {
            this.allocateRowBuffer(rowSize);
            reader.accept(this.rowBuffer);
            this.rowBuffer.flip();
        };
        Iterator<ByteBuffer> it = this.rowDataStore.rowIterator(rowReader, this.rowBuffer);
        return new EntryIterator(it);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reset() {
        Path idxFile;
        ExternalHashTable externalHashTable = this;
        synchronized (externalHashTable) {
            this.checkCancelled();
            idxFile = this.file;
            this.capacity = 16L;
            this.usedSlots = 0;
            this.reusableBuff.clear();
            try {
                this.fileIo.clear();
                this.fileIo.write(this.reusableBuff, this.capacity * 12L);
            }
            catch (IOException e) {
                this.close();
                throw ExternalCollectionUtils.accessFailedException(e);
            }
            this.rowDataStore.reset();
        }
        if (log.isDebugEnabled()) {
            log.debug("External row store file cleaned: " + idxFile.getFileName(), new Object[0]);
        }
    }

    public synchronized int size() {
        this.checkCancelled();
        return this.rowDataStore.size();
    }

    @TestOnly
    public synchronized long lastValueOffset() {
        return this.rowDataStore.lastWrittenOffset();
    }

    private synchronized void insertEntry(int hashCode, ByteBuffer keyBuff, int keyElementCount, ByteBuffer valueBuff) {
        ByteBuffer rowBuffer = this.prepareRowBuffer(keyBuff, keyElementCount, valueBuff);
        this.ensureCapacity();
        long address = this.rowDataStore.write(rowBuffer);
        this.putEntryToFreeSlot(hashCode, address);
    }

    private void modifyOrReplaceEntry(Entry entry, ByteBuffer key, int keyElementCount, ByteBuffer value) {
        if (value.remaining() <= this.tableRowReader.valueBlockSize) {
            this.updateEntry(entry, key, keyElementCount, value);
        } else {
            this.removeEntryAndInsertIntoNewSlot(entry, key, keyElementCount, value);
        }
    }

    private synchronized void updateEntry(Entry entry, ByteBuffer key, int keyElementCount, ByteBuffer value) {
        ByteBuffer rowBuffer = this.prepareRowBuffer(key, keyElementCount, value);
        this.rowDataStore.update(entry.rowAddress, rowBuffer);
    }

    private synchronized void removeEntryAndInsertIntoNewSlot(Entry entry, ByteBuffer key, int keyElementCount, ByteBuffer value) {
        this.removeEntry(entry, entry.hashCode);
        this.insertEntry(entry.hashCode, key, keyElementCount, value);
    }

    private synchronized void removeEntry(Entry entry, int hashCode) {
        this.writeEntryToIndexFile(entry.slot(), hashCode, entry.rowAddress() | Long.MIN_VALUE);
        this.rowDataStore.remove(entry.rowAddress);
    }

    private void putEntryToFreeSlot(int hashCode, long rowAddr) {
        long slot = this.findFreeSlotForInsert(hashCode);
        this.writeEntryToIndexFile(slot, hashCode, rowAddr);
        ++this.usedSlots;
    }

    private long findFreeSlotForInsert(int hashCode) {
        Entry entry;
        long slot;
        long startSlot = slot = this.slot(hashCode);
        while ((entry = this.readEntryFromIndexFile(slot)) != null && !entry.isTombstone() && !entry.isEmpty()) {
            slot = (slot + 1L) % this.capacity;
            assert (slot != startSlot);
        }
        return slot;
    }

    private long slot(long hashCode) {
        long hc64 = hashCode << 48 ^ hashCode << 32 ^ hashCode << 16 ^ hashCode;
        return hc64 & this.capacity - 1L;
    }

    @Nullable
    private Entry findEntry(BinaryTuple key, int hashCode) {
        return this.findEntryAndLoadRow(key, hashCode, false);
    }

    @Nullable
    private Entry findEntryAndLoadRow(BinaryTuple key, int hashCode, boolean loadRow) {
        long slot;
        long initialSlot = slot = this.slot(hashCode);
        Entry entry = this.readEntryFromIndexFile(slot);
        if (loadRow) {
            this.tableRowReader.init(0);
        } else {
            int keyBlockSize = 12 + key.byteBuffer().capacity();
            this.tableRowReader.init(keyBlockSize);
        }
        while (!entry.isEmpty()) {
            if (!entry.isTombstone() && hashCode == entry.hash()) {
                this.rowDataStore.read(entry.rowAddress(), this.tableRowReader);
                BinaryTuple foundRow = this.tableRowReader.getKey();
                assert (foundRow != null);
                if (this.keyComparator.compare(key, foundRow) == 0) break;
            }
            if ((slot = (slot + 1L) % this.capacity) == initialSlot) {
                return null;
            }
            entry = this.readEntryFromIndexFile(slot);
        }
        return entry.isEmpty() || entry.isTombstone() ? null : entry;
    }

    private static ByteBuffer copyValueFromRowBuffer(ByteBuffer rowBuffer, int rowSize) {
        assert (rowSize > 0);
        int keySize = rowBuffer.getInt(4);
        int valueSize = rowBuffer.getInt(8);
        int valueBlockSize = rowSize - 12 - keySize;
        ByteBuffer value = ExternalHashTable.allocateNewBuffer(valueBlockSize);
        rowBuffer.position(12 + keySize);
        rowBuffer.limit(rowBuffer.position() + valueSize);
        value.put(rowBuffer);
        value.flip();
        return value;
    }

    private ByteBuffer allocateRowBuffer(int size) {
        if (this.rowBuffer.capacity() < size) {
            this.rowBuffer = ByteBuffer.allocate(size).order(BinaryTupleParser.ORDER);
        } else {
            this.rowBuffer.clear();
            this.rowBuffer.limit(size);
        }
        return this.rowBuffer;
    }

    private ByteBuffer prepareRowBuffer(ByteBuffer key, int keyElementCount, ByteBuffer value) {
        ByteBuffer rowBuffer = this.allocateRowBuffer(12 + key.remaining() + value.remaining());
        rowBuffer.putInt(keyElementCount);
        rowBuffer.putInt(key.remaining());
        rowBuffer.putInt(value.remaining());
        rowBuffer.put(key);
        rowBuffer.put(value);
        rowBuffer.flip();
        return rowBuffer;
    }

    private Entry readEntryFromIndexFile(long slot) {
        return this.readEntryFromIndexFile(slot, this.fileIo);
    }

    private Entry readEntryFromIndexFile(long slot, FileIo fileCh) {
        try {
            if (slot != -1L) {
                this.gotoSlot(slot);
            } else {
                slot = fileCh.position() / 12L;
            }
            this.reusableBuff.clear();
            fileCh.readFully(this.reusableBuff);
            this.reusableBuff.flip();
            int hashCode = this.reusableBuff.getInt();
            long addr = this.reusableBuff.getLong() - 1L;
            return new Entry(hashCode, addr, slot);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void writeEntryToIndexFile(long slot, int hashCode, long addr) {
        try {
            this.gotoSlot(slot);
            this.reusableBuff.clear();
            this.reusableBuff.putInt(hashCode);
            this.reusableBuff.putLong(addr + 1L);
            this.reusableBuff.flip();
            this.fileIo.writeFully(this.reusableBuff);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void gotoSlot(long slot) {
        try {
            this.fileIo.position(slot * 12L);
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureCapacity() {
        if ((double)this.usedSlots <= 0.5 * (double)this.capacity) {
            return;
        }
        FileIo oldFileIo = this.fileIo;
        File oldIdxFile = this.file.toFile();
        long oldSize = this.capacity;
        try {
            this.initNewIndexFile(oldSize * 2L);
            this.copyDataFromOldFile(oldFileIo, oldSize);
        }
        finally {
            IgniteUtils.closeQuiet((AutoCloseable)oldFileIo);
            this.deleteUnusedFile(oldIdxFile);
        }
    }

    private void copyDataFromOldFile(FileIo oldFile, long oldSize) {
        try {
            this.usedSlots = 0;
            oldFile.position(0L);
            for (long i = 0L; i < oldSize; ++i) {
                Entry e = this.readEntryFromIndexFile(-1L, oldFile);
                if (e.isTombstone() || e.isEmpty()) continue;
                this.putEntryToFreeSlot(e.hash(), e.rowAddress());
            }
        }
        catch (IOException e) {
            IgniteUtils.closeQuiet((AutoCloseable)oldFile);
            IgniteUtils.closeQuiet((AutoCloseable)this);
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initNewIndexFile(long cap) {
        try {
            assert (cap > 0L && (cap & cap - 1L) == 0L) : "cap=" + cap;
            if (cap > 0x400000000000000L) {
                throw new IllegalArgumentException("Maximum capacity is exceeded [curCapacity=" + cap + ", maxCapacity=288230376151711744]");
            }
            Path newFile = this.workDir.createFile("hashIdx");
            ExternalHashTable externalHashTable = this;
            synchronized (externalHashTable) {
                this.capacity = cap;
                this.file = newFile;
                newFile.toFile().deleteOnExit();
                this.fileIo = this.fileIoFactory.create(this.file, new OpenOption[]{StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE});
                this.reusableBuff.clear();
                this.fileIo.write(this.reusableBuff, cap * 12L);
            }
        }
        catch (IOException e) {
            this.close();
            throw ExternalCollectionUtils.accessFailedException(e);
        }
    }

    private void checkCancelled() {
        if (this.isClosed) {
            throw new SqlException(GridgainErrorGroups.MemoryQuota.SPILLING_ERR, "Row store has been closed.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        ExternalHashTable externalHashTable = this;
        synchronized (externalHashTable) {
            if (this.isClosed) {
                return;
            }
            this.isClosed = true;
        }
        IgniteUtils.closeQuiet((AutoCloseable)this.fileIo);
        this.rowDataStore.close();
        this.deleteUnusedFile(this.file.toFile());
    }

    private void deleteUnusedFile(File file) {
        if (!file.delete()) {
            log.info("Failed to remove spill file " + file.getName(), new Object[0]);
        } else if (log.isDebugEnabled()) {
            log.debug("Spill file removed " + file.getName(), new Object[0]);
        }
    }

    private static ByteBuffer allocateNewBuffer(int capacity) {
        return ByteBuffer.allocate(capacity).order(BinaryTupleParser.ORDER);
    }

    private class ExternalRowReader
    implements ExternalFileStore.RowReader {
        private int bufSize;
        private int keyBlockSize;
        private int valueBlockSize;
        private int rowSize;

        private ExternalRowReader() {
        }

        void init(int bufSize) {
            this.rowSize = -1;
            this.keyBlockSize = -1;
            this.valueBlockSize = -1;
            if (bufSize != 0) {
                ExternalHashTable.this.allocateRowBuffer(bufSize);
            }
            this.bufSize = bufSize;
        }

        @Override
        public void readRow(int rowSize, Consumer<ByteBuffer> reader) {
            if (this.bufSize == 0) {
                ExternalHashTable.this.allocateRowBuffer(rowSize);
            }
            reader.accept(ExternalHashTable.this.rowBuffer);
            ExternalHashTable.this.rowBuffer.flip();
            this.rowSize = rowSize;
        }

        private BinaryTuple getKey() {
            int keyElementsNum = ExternalHashTable.this.rowBuffer.getInt(0);
            this.keyBlockSize = ExternalHashTable.this.rowBuffer.getInt(4);
            this.valueBlockSize = ExternalHashTable.this.rowBuffer.getInt(8);
            ExternalHashTable.this.rowBuffer.position(12);
            ExternalHashTable.this.rowBuffer.limit(12 + this.keyBlockSize);
            ByteBuffer keyBlock = ExternalHashTable.this.rowBuffer.slice().order(BinaryTupleParser.ORDER);
            ExternalHashTable.this.rowBuffer.position(0);
            if (this.bufSize == 0) {
                ExternalHashTable.this.rowBuffer.limit(12 + this.keyBlockSize + this.valueBlockSize);
            }
            return new BinaryTuple(keyElementsNum, keyBlock);
        }
    }

    private static class Entry {
        private final int hashCode;
        private final long rowAddress;
        private final long slot;

        Entry(int hashCode, long rowAddress, long slot) {
            this.hashCode = hashCode;
            this.rowAddress = rowAddress;
            this.slot = slot;
        }

        int hash() {
            return this.hashCode;
        }

        long rowAddress() {
            return this.rowAddress & Long.MAX_VALUE;
        }

        long slot() {
            return this.slot;
        }

        boolean isTombstone() {
            return (this.rowAddress & Long.MIN_VALUE) != 0L;
        }

        boolean isEmpty() {
            return this.rowAddress == -1L;
        }
    }

    private class EntryIterator
    implements Iterator<Map.Entry<BinaryTuple, ByteBuffer>> {
        private final Iterator<ByteBuffer> it;
        private BinaryTuple key;

        private EntryIterator(Iterator<ByteBuffer> it) {
            this.it = it;
        }

        @Override
        public boolean hasNext() {
            return this.it.hasNext();
        }

        @Override
        public Map.Entry<BinaryTuple, ByteBuffer> next() {
            ByteBuffer rowBuff = this.it.next();
            int keyElementsNum = rowBuff.getInt(0);
            int keyBlockSize = rowBuff.getInt(4);
            int valueBlockSize = rowBuff.getInt(8);
            ByteBuffer keyBuff = ExternalHashTable.allocateNewBuffer(keyBlockSize);
            rowBuff.position(12);
            rowBuff.limit(12 + keyBlockSize);
            keyBuff.put(rowBuff);
            keyBuff.flip();
            this.key = new BinaryTuple(keyElementsNum, keyBuff);
            ByteBuffer valueBuf = ExternalHashTable.allocateNewBuffer(valueBlockSize);
            rowBuff.position(12 + keyBlockSize);
            rowBuff.limit(12 + keyBlockSize + valueBlockSize);
            valueBuf.put(rowBuff);
            valueBuf.flip();
            return Map.entry(this.key, valueBuf);
        }

        @Override
        public void remove() {
            if (this.key == null) {
                throw new IllegalStateException();
            }
            ExternalHashTable.this.remove(this.key);
            this.key = null;
        }
    }
}

