/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.raft.storage.segstore;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.raft.storage.segstore.GroupIndexMeta;
import org.apache.ignite.internal.raft.storage.segstore.IndexFileMeta;
import org.apache.ignite.internal.raft.storage.segstore.ReadModeIndexMemTable;
import org.apache.ignite.internal.raft.storage.segstore.SegmentFilePointer;
import org.apache.ignite.internal.raft.storage.segstore.SegmentInfo;
import org.apache.ignite.internal.util.IgniteUtils;
import org.jetbrains.annotations.Nullable;

class IndexFileManager {
    private static final IgniteLogger LOG = Loggers.forClass(IndexFileManager.class);
    static final int MAGIC_NUMBER = 1810933610;
    static final int FORMAT_VERSION = 1;
    private static final String INDEX_FILE_NAME_FORMAT = "index-%010d-%010d.bin";
    private static final Pattern INDEX_FILE_NAME_PATTERN = Pattern.compile("index-(?<ordinal>\\d{10})-(?<generation>\\d{10})\\.bin");
    private static final String TMP_FILE_SUFFIX = ".tmp";
    static final int SEGMENT_FILE_OFFSET_SIZE = 4;
    static final int COMMON_META_SIZE = 12;
    static final int GROUP_META_SIZE = 40;
    static final ByteOrder BYTE_ORDER = ByteOrder.LITTLE_ENDIAN;
    private final Path indexFilesDir;
    private int curFileOrdinal = -1;
    private final Map<Long, GroupIndexMeta> groupIndexMetas = new ConcurrentHashMap<Long, GroupIndexMeta>();

    IndexFileManager(Path baseDir) throws IOException {
        this.indexFilesDir = baseDir.resolve("index");
        Files.createDirectories(this.indexFilesDir, new FileAttribute[0]);
    }

    void start() throws IOException {
        try (Stream<Path> indexFiles = Files.list(this.indexFilesDir);){
            Iterator it = indexFiles.sorted().iterator();
            while (it.hasNext()) {
                this.recoverIndexFileMetas((Path)it.next());
            }
        }
    }

    Path indexFilesDir() {
        return this.indexFilesDir;
    }

    void cleanupTmpFiles() throws IOException {
        try (Stream<Path> indexFiles = Files.list(this.indexFilesDir);){
            Iterator it = indexFiles.iterator();
            while (it.hasNext()) {
                Path indexFile = (Path)it.next();
                if (!indexFile.getFileName().toString().endsWith(TMP_FILE_SUFFIX)) continue;
                LOG.info("Deleting temporary index file: {}.", new Object[]{indexFile});
                Files.delete(indexFile);
            }
        }
    }

    Path saveIndexMemtable(ReadModeIndexMemTable indexMemTable) throws IOException {
        return this.saveIndexMemtable(indexMemTable, ++this.curFileOrdinal, false);
    }

    private Path saveIndexMemtable(ReadModeIndexMemTable indexMemTable, int fileOrdinal, boolean onRecovery) throws IOException {
        String fileName = IndexFileManager.indexFileName(fileOrdinal, 0);
        Path tmpFilePath = this.indexFilesDir.resolve(fileName + TMP_FILE_SUFFIX);
        try (BufferedOutputStream os = new BufferedOutputStream(Files.newOutputStream(tmpFilePath, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE));){
            byte[] headerBytes = this.serializeHeaderAndFillMetadata(indexMemTable, fileOrdinal, onRecovery);
            os.write(headerBytes);
            Iterator<Map.Entry<Long, SegmentInfo>> it = indexMemTable.iterator();
            while (it.hasNext()) {
                SegmentInfo segmentInfo = it.next().getValue();
                if (segmentInfo.size() <= 0) continue;
                os.write(IndexFileManager.payload(segmentInfo));
            }
        }
        return IndexFileManager.syncAndRename(tmpFilePath, tmpFilePath.resolveSibling(fileName));
    }

    void recoverIndexFile(ReadModeIndexMemTable indexMemTable, int fileOrdinal) throws IOException {
        this.saveIndexMemtable(indexMemTable, fileOrdinal, true);
    }

    @Nullable
    SegmentFilePointer getSegmentFilePointer(long groupId, long logIndex) throws IOException {
        GroupIndexMeta groupIndexMeta = this.groupIndexMetas.get(groupId);
        if (groupIndexMeta == null) {
            return null;
        }
        IndexFileMeta indexFileMeta = groupIndexMeta.indexMeta(logIndex);
        if (indexFileMeta == null) {
            return null;
        }
        Path indexFile = this.indexFilesDir.resolve(IndexFileManager.indexFileName(indexFileMeta.indexFileOrdinal(), 0));
        long payloadArrayIndex = logIndex - indexFileMeta.firstLogIndexInclusive();
        assert (payloadArrayIndex >= 0L) : payloadArrayIndex;
        long payloadOffset = (long)indexFileMeta.indexFilePayloadOffset() + payloadArrayIndex * 4L;
        try (SeekableByteChannel channel = Files.newByteChannel(indexFile, StandardOpenOption.READ);){
            channel.position(payloadOffset);
            ByteBuffer segmentPayloadOffsetBuffer = ByteBuffer.allocate(4).order(BYTE_ORDER);
            while (segmentPayloadOffsetBuffer.hasRemaining()) {
                int bytesRead = channel.read(segmentPayloadOffsetBuffer);
                if (bytesRead != -1) continue;
                throw new EOFException("EOF reached while reading index file: " + indexFile);
            }
            int segmentPayloadOffset = segmentPayloadOffsetBuffer.getInt(0);
            SegmentFilePointer segmentFilePointer = new SegmentFilePointer(indexFileMeta.indexFileOrdinal(), segmentPayloadOffset);
            return segmentFilePointer;
        }
    }

    long firstLogIndexInclusive(long groupId) {
        GroupIndexMeta groupIndexMeta = this.groupIndexMetas.get(groupId);
        return groupIndexMeta == null ? -1L : groupIndexMeta.firstLogIndexInclusive();
    }

    long lastLogIndexExclusive(long groupId) {
        GroupIndexMeta groupIndexMeta = this.groupIndexMetas.get(groupId);
        return groupIndexMeta == null ? -1L : groupIndexMeta.lastLogIndexExclusive();
    }

    boolean indexFileExists(int fileOrdinal) {
        return Files.exists(this.indexFilesDir.resolve(IndexFileManager.indexFileName(fileOrdinal, 0)), new LinkOption[0]);
    }

    private byte[] serializeHeaderAndFillMetadata(ReadModeIndexMemTable indexMemTable, int fileOrdinal, boolean onRecovery) {
        int numGroups = indexMemTable.numGroups();
        int headerSize = IndexFileManager.headerSize(numGroups);
        ByteBuffer headerBuffer = ByteBuffer.allocate(headerSize).order(BYTE_ORDER).putInt(1810933610).putInt(1).putInt(numGroups);
        int payloadOffset = headerSize;
        Iterator<Map.Entry<Long, SegmentInfo>> it = indexMemTable.iterator();
        while (it.hasNext()) {
            Map.Entry<Long, SegmentInfo> entry = it.next();
            Long groupId = entry.getKey();
            SegmentInfo segmentInfo = entry.getValue();
            long firstLogIndexInclusive = segmentInfo.firstLogIndexInclusive();
            long lastLogIndexExclusive = segmentInfo.lastLogIndexExclusive();
            long firstIndexKept = segmentInfo.firstIndexKept();
            if (!onRecovery) {
                IndexFileMeta indexFileMeta = IndexFileManager.createIndexFileMeta(firstLogIndexInclusive, lastLogIndexExclusive, firstIndexKept, payloadOffset, fileOrdinal);
                this.putIndexFileMeta(groupId, indexFileMeta, firstIndexKept);
            }
            headerBuffer.putLong(groupId).putInt(0).putInt(payloadOffset).putLong(firstLogIndexInclusive).putLong(lastLogIndexExclusive).putLong(firstIndexKept);
            payloadOffset += IndexFileManager.payloadSize(segmentInfo);
        }
        return headerBuffer.array();
    }

    @Nullable
    private static IndexFileMeta createIndexFileMeta(long firstLogIndexInclusive, long lastLogIndexExclusive, long firstIndexKept, int payloadOffset, int fileOrdinal) {
        if (firstLogIndexInclusive == -1L) {
            assert (firstIndexKept != -1L) : "Expected a prefix tombstone, but firstIndexKept is not set.";
            return null;
        }
        if (firstIndexKept == -1L || firstIndexKept <= firstLogIndexInclusive) {
            return new IndexFileMeta(firstLogIndexInclusive, lastLogIndexExclusive, payloadOffset, fileOrdinal);
        }
        int numEntriesToSkip = Math.toIntExact(firstIndexKept - firstLogIndexInclusive);
        int adjustedPayloadOffset = payloadOffset + numEntriesToSkip * 4;
        return new IndexFileMeta(firstIndexKept, lastLogIndexExclusive, adjustedPayloadOffset, fileOrdinal);
    }

    private void putIndexFileMeta(Long groupId, @Nullable IndexFileMeta indexFileMeta, long firstIndexKept) {
        GroupIndexMeta existingGroupIndexMeta = this.groupIndexMetas.get(groupId);
        if (existingGroupIndexMeta == null) {
            if (indexFileMeta != null) {
                this.groupIndexMetas.put(groupId, new GroupIndexMeta(indexFileMeta));
            }
        } else {
            if (firstIndexKept != -1L) {
                existingGroupIndexMeta.truncatePrefix(firstIndexKept);
            }
            if (indexFileMeta != null) {
                assert (indexFileMeta.firstLogIndexInclusive() >= firstIndexKept) : indexFileMeta;
                existingGroupIndexMeta.addIndexMeta(indexFileMeta);
            }
        }
    }

    private static Path syncAndRename(Path from, Path to) throws IOException {
        IgniteUtils.fsyncFile((Path)from);
        return IgniteUtils.atomicMoveFile((Path)from, (Path)to, (IgniteLogger)LOG);
    }

    private static byte[] payload(SegmentInfo segmentInfo) {
        ByteBuffer payloadBuffer = ByteBuffer.allocate(IndexFileManager.payloadSize(segmentInfo)).order(BYTE_ORDER);
        segmentInfo.saveOffsetsTo(payloadBuffer);
        return payloadBuffer.array();
    }

    private static int headerSize(int numGroups) {
        return 12 + numGroups * 40;
    }

    private static int payloadSize(SegmentInfo segmentInfo) {
        return segmentInfo.size() * 4;
    }

    private static String indexFileName(int fileOrdinal, int generation) {
        return String.format(INDEX_FILE_NAME_FORMAT, fileOrdinal, generation);
    }

    private void recoverIndexFileMetas(Path indexFile) throws IOException {
        int fileOrdinal = IndexFileManager.indexFileOrdinal(indexFile);
        if (this.curFileOrdinal >= 0 && fileOrdinal != this.curFileOrdinal + 1) {
            throw new IllegalStateException(String.format("Unexpected index file ordinal. Expected %d, actual %d (%s).", this.curFileOrdinal + 1, fileOrdinal, indexFile));
        }
        this.curFileOrdinal = fileOrdinal;
        try (BufferedInputStream is = new BufferedInputStream(Files.newInputStream(indexFile, StandardOpenOption.READ));){
            ByteBuffer commonMetaBuffer = IndexFileManager.readBytes(is, 12, indexFile);
            int magicNumber = commonMetaBuffer.getInt();
            if (magicNumber != 1810933610) {
                throw new IllegalStateException(String.format("Invalid magic number in index file %s: %d.", indexFile, magicNumber));
            }
            int formatVersion = commonMetaBuffer.getInt();
            if (formatVersion > 1) {
                throw new IllegalStateException(String.format("Unsupported format version in index file %s: %d.", indexFile, formatVersion));
            }
            int numGroups = commonMetaBuffer.getInt();
            if (numGroups <= 0) {
                throw new IllegalStateException(String.format("Unexpected number of groups in index file %s: %d.", indexFile, numGroups));
            }
            for (int i = 0; i < numGroups; ++i) {
                ByteBuffer groupMetaBuffer = IndexFileManager.readBytes(is, 40, indexFile);
                long groupId = groupMetaBuffer.getLong();
                groupMetaBuffer.getInt();
                int payloadOffset = groupMetaBuffer.getInt();
                long firstLogIndexInclusive = groupMetaBuffer.getLong();
                long lastLogIndexExclusive = groupMetaBuffer.getLong();
                long firstIndexKept = groupMetaBuffer.getLong();
                IndexFileMeta indexFileMeta = IndexFileManager.createIndexFileMeta(firstLogIndexInclusive, lastLogIndexExclusive, firstIndexKept, payloadOffset, fileOrdinal);
                this.putIndexFileMeta(groupId, indexFileMeta, firstIndexKept);
            }
        }
    }

    private static int indexFileOrdinal(Path indexFile) {
        String fileName = indexFile.getFileName().toString();
        Matcher matcher = INDEX_FILE_NAME_PATTERN.matcher(fileName);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid index file name format: %s.", indexFile));
        }
        return Integer.parseInt(matcher.group("ordinal"));
    }

    private static ByteBuffer readBytes(InputStream is, int size, Path indexFile) throws IOException {
        ByteBuffer result = ByteBuffer.wrap(is.readNBytes(size)).order(BYTE_ORDER);
        if (result.remaining() != size) {
            throw new IOException(String.format("Unexpected EOF when trying to read from index file: %s.", indexFile));
        }
        return result;
    }
}

