/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.raft.storage.segstore;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.ignite.internal.close.ManuallyCloseable;
import org.apache.ignite.internal.failure.FailureProcessor;
import org.apache.ignite.internal.lang.IgniteInternalException;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.raft.storage.segstore.IndexFileManager;
import org.apache.ignite.internal.raft.storage.segstore.IndexMemTable;
import org.apache.ignite.internal.raft.storage.segstore.RaftLogCheckpointer;
import org.apache.ignite.internal.raft.storage.segstore.SegmentFile;
import org.apache.ignite.internal.raft.storage.segstore.SegmentFilePointer;
import org.apache.ignite.internal.raft.storage.segstore.SegmentFileWithMemtable;
import org.apache.ignite.internal.raft.storage.segstore.SegmentInfo;
import org.apache.ignite.internal.raft.storage.segstore.SegmentPayload;
import org.apache.ignite.internal.raft.storage.segstore.WriteBufferWithMemtable;
import org.apache.ignite.internal.raft.storage.segstore.WriteModeIndexMemTable;
import org.apache.ignite.internal.raft.util.VarlenEncoder;
import org.apache.ignite.internal.util.FastCrc;
import org.apache.ignite.lang.ErrorGroups;
import org.apache.ignite.raft.jraft.entity.LogEntry;
import org.apache.ignite.raft.jraft.entity.codec.LogEntryDecoder;
import org.apache.ignite.raft.jraft.entity.codec.LogEntryEncoder;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

class SegmentFileManager
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(SegmentFileManager.class);
    private static final int ROLLOVER_WAIT_TIMEOUT_MS = 30000;
    private static final int MAGIC_NUMBER = 1457567014;
    private static final int FORMAT_VERSION = 1;
    private static final String SEGMENT_FILE_NAME_FORMAT = "segment-%010d-%010d.bin";
    private static final Pattern SEGMENT_FILE_NAME_PATTERN = Pattern.compile("segment-(?<ordinal>\\d{10})-(?<generation>\\d{10})\\.bin");
    static final byte[] HEADER_RECORD = ByteBuffer.allocate(8).order(SegmentFile.BYTE_ORDER).putInt(1457567014).putInt(1).array();
    static final byte[] SWITCH_SEGMENT_RECORD = new byte[8];
    private final Path segmentFilesDir;
    private final long fileSize;
    private final int stripes;
    private final AtomicReference<SegmentFileWithMemtable> currentSegmentFile = new AtomicReference();
    private final RaftLogCheckpointer checkpointer;
    private final IndexFileManager indexFileManager;
    private final Object rolloverLock = new Object();
    private volatile int curSegmentFileOrdinal;
    private boolean isStopped;

    SegmentFileManager(String nodeName, Path baseDir, long fileSize, int stripes, FailureProcessor failureProcessor) throws IOException {
        if (fileSize <= (long)HEADER_RECORD.length) {
            throw new IllegalArgumentException("File size must be greater than the header size: " + fileSize);
        }
        this.segmentFilesDir = baseDir.resolve("segments");
        Files.createDirectories(this.segmentFilesDir, new FileAttribute[0]);
        this.fileSize = fileSize;
        this.stripes = stripes;
        this.indexFileManager = new IndexFileManager(baseDir);
        this.checkpointer = new RaftLogCheckpointer(nodeName, this.indexFileManager, failureProcessor);
    }

    void start() throws IOException {
        LOG.info("Starting segment file manager [segmentFilesDir={}, fileSize={}].", new Object[]{this.segmentFilesDir, this.fileSize});
        this.indexFileManager.cleanupTmpFiles();
        Path lastSegmentFilePath = null;
        try (Stream<Path> segmentFiles = Files.list(this.segmentFilesDir);){
            Iterator it = segmentFiles.sorted().iterator();
            while (it.hasNext()) {
                Path segmentFilePath = (Path)it.next();
                if (!it.hasNext()) {
                    lastSegmentFilePath = segmentFilePath;
                    continue;
                }
                int segmentFileOrdinal = SegmentFileManager.segmentFileOrdinal(segmentFilePath);
                if (this.indexFileManager.indexFileExists(segmentFileOrdinal)) continue;
                LOG.info("Creating missing index file for segment file {}.", new Object[]{segmentFilePath});
                SegmentFile segmentFile = SegmentFile.openExisting(segmentFilePath);
                WriteModeIndexMemTable memTable = this.recoverMemtable(segmentFile, segmentFilePath);
                this.indexFileManager.saveIndexMemtable(memTable.transitionToReadMode(), segmentFileOrdinal);
            }
        }
        if (lastSegmentFilePath == null) {
            this.currentSegmentFile.set(this.allocateNewSegmentFile(0));
        } else {
            this.curSegmentFileOrdinal = SegmentFileManager.segmentFileOrdinal(lastSegmentFilePath);
            this.currentSegmentFile.set(this.recoverLatestSegmentFile(lastSegmentFilePath));
        }
        LOG.info("Segment file manager recovery completed. Current segment file: {}.", new Object[]{lastSegmentFilePath});
        this.indexFileManager.start();
        this.checkpointer.start();
    }

    Path segmentFilesDir() {
        return this.segmentFilesDir;
    }

    Path indexFilesDir() {
        return this.indexFileManager.indexFilesDir();
    }

    @TestOnly
    IndexFileManager indexFileManager() {
        return this.indexFileManager;
    }

    private SegmentFileWithMemtable allocateNewSegmentFile(int fileOrdinal) throws IOException {
        Path path = this.segmentFilesDir.resolve(SegmentFileManager.segmentFileName(fileOrdinal, 0));
        SegmentFile segmentFile = SegmentFile.createNew(path, this.fileSize);
        SegmentFileManager.writeHeader(segmentFile);
        return new SegmentFileWithMemtable(segmentFile, new IndexMemTable(this.stripes), false);
    }

    private SegmentFileWithMemtable recoverLatestSegmentFile(Path segmentFilePath) throws IOException {
        SegmentFile segmentFile = SegmentFile.openExisting(segmentFilePath);
        return new SegmentFileWithMemtable(segmentFile, this.recoverLatestMemtable(segmentFile, segmentFilePath), false);
    }

    private static String segmentFileName(int fileOrdinal, int generation) {
        return String.format(SEGMENT_FILE_NAME_FORMAT, fileOrdinal, generation);
    }

    private static SegmentFileWithMemtable convertToReadOnly(SegmentFileWithMemtable segmentFile) {
        return new SegmentFileWithMemtable(segmentFile.segmentFile(), segmentFile.memtable(), true);
    }

    void appendEntry(long groupId, LogEntry entry, LogEntryEncoder encoder) throws IOException {
        int segmentEntrySize = SegmentPayload.size(entry, encoder);
        if ((long)segmentEntrySize > this.maxPossibleEntrySize()) {
            throw new IllegalArgumentException(String.format("Segment entry is too big (%d bytes), maximum allowed segment entry size: %d bytes.", segmentEntrySize, this.maxPossibleEntrySize()));
        }
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(segmentEntrySize);){
            ByteBuffer segmentBuffer = writeBufferWithMemtable.buffer();
            int segmentOffset = segmentBuffer.position();
            SegmentPayload.writeTo(segmentBuffer, groupId, segmentEntrySize, entry, encoder);
            writeBufferWithMemtable.memtable().appendSegmentFileOffset(groupId, entry.getId().getIndex(), segmentOffset);
        }
    }

    @Nullable
    LogEntry getEntry(long groupId, long logIndex, LogEntryDecoder decoder) throws IOException {
        ByteBuffer entryBuffer = this.getEntry(groupId, logIndex);
        return entryBuffer == null ? null : SegmentPayload.readFrom(entryBuffer, decoder);
    }

    @Nullable
    private ByteBuffer getEntry(long groupId, long logIndex) throws IOException {
        ByteBuffer bufferFromCheckpointQueue;
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        if (segmentInfo != null) {
            if (logIndex >= segmentInfo.lastLogIndexExclusive()) {
                return null;
            }
            int segmentPayloadOffset = segmentInfo.getOffset(logIndex);
            if (segmentPayloadOffset != 0) {
                return currentSegmentFile.segmentFile().buffer().position(segmentPayloadOffset);
            }
        }
        if ((bufferFromCheckpointQueue = this.checkpointer.findSegmentPayloadInQueue(groupId, logIndex)) != null) {
            return bufferFromCheckpointQueue;
        }
        return this.readFromOtherSegmentFiles(groupId, logIndex);
    }

    void truncateSuffix(long groupId, long lastLogIndexKept) throws IOException {
        try (WriteBufferWithMemtable writeBufferWithMemtable = this.reserveBytesWithRollover(24);){
            ByteBuffer segmentBuffer = writeBufferWithMemtable.buffer();
            SegmentPayload.writeTruncateSuffixRecordTo(segmentBuffer, groupId, lastLogIndexKept);
            writeBufferWithMemtable.memtable().truncateSuffix(groupId, lastLogIndexKept);
        }
    }

    private WriteBufferWithMemtable reserveBytesWithRollover(int size) throws IOException {
        SegmentFileWithMemtable segmentFileWithMemtable;
        SegmentFile.WriteBuffer writeBuffer;
        while ((writeBuffer = (segmentFileWithMemtable = this.currentSegmentFile()).segmentFile().reserve(size)) == null) {
            this.initiateRollover(segmentFileWithMemtable);
        }
        return new WriteBufferWithMemtable(writeBuffer, segmentFileWithMemtable.memtable());
    }

    long firstLogIndexInclusive(long groupId) {
        long logIndexFromMemtable = this.firstLogIndexFromMemtable(groupId);
        long logIndexFromCheckpointQueue = this.checkpointer.firstLogIndexInclusive(groupId);
        long logIndexFromIndexFiles = this.indexFileManager.firstLogIndexInclusive(groupId);
        if (logIndexFromIndexFiles >= 0L) {
            return logIndexFromIndexFiles;
        }
        if (logIndexFromCheckpointQueue >= 0L) {
            return logIndexFromCheckpointQueue;
        }
        return logIndexFromMemtable;
    }

    private long firstLogIndexFromMemtable(long groupId) {
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        if (segmentInfo == null || segmentInfo.size() == 0) {
            return -1L;
        }
        return segmentInfo.firstLogIndexInclusive();
    }

    long lastLogIndexExclusive(long groupId) {
        long logIndexFromMemtable = this.lastLogIndexFromMemtable(groupId);
        if (logIndexFromMemtable >= 0L) {
            return logIndexFromMemtable;
        }
        long logIndexFromCheckpointQueue = this.checkpointer.lastLogIndexExclusive(groupId);
        if (logIndexFromCheckpointQueue >= 0L) {
            return logIndexFromCheckpointQueue;
        }
        return this.indexFileManager.lastLogIndexExclusive(groupId);
    }

    private long lastLogIndexFromMemtable(long groupId) {
        SegmentFileWithMemtable currentSegmentFile = this.currentSegmentFile.get();
        SegmentInfo segmentInfo = currentSegmentFile.memtable().segmentInfo(groupId);
        return segmentInfo == null ? -1L : segmentInfo.lastLogIndexExclusive();
    }

    private SegmentFileWithMemtable currentSegmentFile() {
        SegmentFileWithMemtable segmentFile = this.currentSegmentFile.get();
        if (!segmentFile.readOnly()) {
            return segmentFile;
        }
        try {
            Object object = this.rolloverLock;
            synchronized (object) {
                while (true) {
                    if (this.isStopped) {
                        throw new IgniteInternalException(ErrorGroups.Common.NODE_STOPPING_ERR);
                    }
                    segmentFile = this.currentSegmentFile.get();
                    if (!segmentFile.readOnly()) {
                        return segmentFile;
                    }
                    this.rolloverLock.wait(30000L);
                }
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, "Interrupted while waiting for rollover.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initiateRollover(SegmentFileWithMemtable observedSegmentFile) throws IOException {
        if (!this.currentSegmentFile.compareAndSet(observedSegmentFile, SegmentFileManager.convertToReadOnly(observedSegmentFile))) {
            return;
        }
        this.checkpointer.onRollover(observedSegmentFile.segmentFile(), observedSegmentFile.memtable().transitionToReadMode());
        Object object = this.rolloverLock;
        synchronized (object) {
            if (this.isStopped) {
                throw new IgniteInternalException(ErrorGroups.Common.NODE_STOPPING_ERR);
            }
            this.currentSegmentFile.set(this.allocateNewSegmentFile(++this.curSegmentFileOrdinal));
            this.rolloverLock.notifyAll();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws Exception {
        Object object = this.rolloverLock;
        synchronized (object) {
            if (this.isStopped) {
                return;
            }
            this.isStopped = true;
            SegmentFileWithMemtable segmentFile = this.currentSegmentFile.get();
            if (segmentFile != null) {
                segmentFile.segmentFile().close();
            }
            this.rolloverLock.notifyAll();
        }
        this.checkpointer.stop();
    }

    private static void writeHeader(SegmentFile segmentFile) {
        try (SegmentFile.WriteBuffer writeBuffer = segmentFile.reserve(HEADER_RECORD.length);){
            assert (writeBuffer != null);
            writeBuffer.buffer().put(HEADER_RECORD);
        }
    }

    private long maxPossibleEntrySize() {
        return this.fileSize - (long)HEADER_RECORD.length;
    }

    @Nullable
    private ByteBuffer readFromOtherSegmentFiles(long groupId, long logIndex) throws IOException {
        SegmentFilePointer segmentFilePointer = this.indexFileManager.getSegmentFilePointer(groupId, logIndex);
        if (segmentFilePointer == null) {
            return null;
        }
        Path path = this.segmentFilesDir.resolve(SegmentFileManager.segmentFileName(segmentFilePointer.fileOrdinal(), 0));
        SegmentFile segmentFile = SegmentFile.openExisting(path);
        return segmentFile.buffer().position(segmentFilePointer.payloadOffset());
    }

    private WriteModeIndexMemTable recoverMemtable(SegmentFile segmentFile, Path segmentFilePath) {
        ByteBuffer buffer = segmentFile.buffer();
        SegmentFileManager.validateSegmentFileHeader(buffer, segmentFilePath);
        IndexMemTable memtable = new IndexMemTable(this.stripes);
        while (buffer.remaining() > SWITCH_SEGMENT_RECORD.length) {
            int segmentFilePayloadOffset = buffer.position();
            long groupId = buffer.getLong();
            int payloadLength = buffer.getInt();
            if (payloadLength == 0) {
                long lastLogIndexKept = buffer.getLong();
                memtable.truncateSuffix(groupId, lastLogIndexKept);
                buffer.position(buffer.position() + 4);
                continue;
            }
            int endOfRecordPosition = buffer.position() + payloadLength + 4;
            long index = VarlenEncoder.readLong(buffer);
            memtable.appendSegmentFileOffset(groupId, index, segmentFilePayloadOffset);
            buffer.position(endOfRecordPosition);
        }
        return memtable;
    }

    private WriteModeIndexMemTable recoverLatestMemtable(SegmentFile segmentFile, Path segmentFilePath) {
        ByteBuffer buffer = segmentFile.buffer();
        SegmentFileManager.validateSegmentFileHeader(buffer, segmentFilePath);
        IndexMemTable memtable = new IndexMemTable(this.stripes);
        while (buffer.remaining() > SWITCH_SEGMENT_RECORD.length) {
            int crcPosition;
            int segmentFilePayloadOffset = buffer.position();
            long groupId = buffer.getLong();
            int payloadLength = buffer.getInt();
            if (payloadLength == 0) {
                long lastLogIndexKept = buffer.getLong();
                crcPosition = buffer.position();
                buffer.position(segmentFilePayloadOffset);
                if (!SegmentFileManager.isCrcValid(buffer, crcPosition)) break;
                memtable.truncateSuffix(groupId, lastLogIndexKept);
            } else {
                crcPosition = buffer.position() + payloadLength;
                long index = VarlenEncoder.readLong(buffer);
                buffer.position(segmentFilePayloadOffset);
                if (!SegmentFileManager.isCrcValid(buffer, crcPosition)) break;
                memtable.appendSegmentFileOffset(groupId, index, segmentFilePayloadOffset);
            }
            buffer.position(crcPosition + 4);
        }
        return memtable;
    }

    private static boolean isCrcValid(ByteBuffer buffer, int crcPosition) {
        int originalPosition = buffer.position();
        int crc = buffer.getInt(crcPosition);
        int expectedCrc = FastCrc.calcCrc((ByteBuffer)buffer, (int)(crcPosition - buffer.position()));
        buffer.position(originalPosition);
        return crc == expectedCrc;
    }

    private static void validateSegmentFileHeader(ByteBuffer buffer, Path segmentFilePath) {
        int magicNumber = buffer.getInt();
        if (magicNumber != 1457567014) {
            throw new IllegalStateException(String.format("Invalid magic number in segment file %s: %d.", segmentFilePath, magicNumber));
        }
        int formatVersion = buffer.getInt();
        if (formatVersion > 1) {
            throw new IllegalStateException(String.format("Unsupported format version in segment file %s: %d.", segmentFilePath, formatVersion));
        }
    }

    private static int segmentFileOrdinal(Path segmentFile) {
        String fileName = segmentFile.getFileName().toString();
        Matcher matcher = SEGMENT_FILE_NAME_PATTERN.matcher(fileName);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid segment file name format: %s.", segmentFile));
        }
        return Integer.parseInt(matcher.group("ordinal"));
    }
}

