/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.ml.util.generators.primitives.vector;

import java.util.Arrays;
import java.util.List;
import org.apache.ignite.internal.util.typedef.internal.A;
import org.apache.ignite.ml.math.functions.IgniteFunction;
import org.apache.ignite.ml.math.primitives.vector.Vector;
import org.apache.ignite.ml.math.primitives.vector.VectorUtils;
import org.apache.ignite.ml.util.generators.primitives.scalar.RandomProducer;

/**
 * Generate vectors having components generated by parametrized function.
 * For each vector v: v = [F1(t), F2(t), ..., Fn(t)], where t is a value
 * from user defined distribution.
 *
 */
public class ParametricVectorGenerator implements VectorGenerator {
    /** Per dimension generators. */
    private final List<IgniteFunction<Double, Double>> perDimensionGenerators;

    /** Random producer. */
    private final RandomProducer randomProducer;

    /**
     * Create an intance of ParametricVectorGenerator.
     *
     * @param paramGenerator Parameter generator.
     * @param perDimensionGenerators Per dimension generators.
     */
    public ParametricVectorGenerator(RandomProducer paramGenerator,
        IgniteFunction<Double, Double>... perDimensionGenerators) {

        A.notEmpty(perDimensionGenerators, "perDimensionGenerators.length != 0");

        this.perDimensionGenerators = Arrays.asList(perDimensionGenerators);
        this.randomProducer = paramGenerator;
    }

    /** {@inheritDoc} */
    @Override public Vector get() {
        Double t = randomProducer.get();
        return VectorUtils.of(perDimensionGenerators.stream()
            .mapToDouble(f -> f.apply(t)).toArray());
    }
}
