/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.index;

import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.apache.ignite.internal.catalog.Catalog;
import org.apache.ignite.internal.catalog.CatalogService;
import org.apache.ignite.internal.catalog.descriptors.CatalogIndexDescriptor;
import org.apache.ignite.internal.catalog.descriptors.CatalogIndexStatus;
import org.apache.ignite.internal.catalog.descriptors.CatalogTableDescriptor;
import org.apache.ignite.internal.catalog.descriptors.CatalogZoneDescriptor;
import org.apache.ignite.internal.catalog.events.CatalogEvent;
import org.apache.ignite.internal.catalog.events.RemoveIndexEventParameters;
import org.apache.ignite.internal.catalog.events.StartBuildingIndexEventParameters;
import org.apache.ignite.internal.close.ManuallyCloseable;
import org.apache.ignite.internal.event.Event;
import org.apache.ignite.internal.event.EventListener;
import org.apache.ignite.internal.failure.FailureContext;
import org.apache.ignite.internal.failure.FailureProcessor;
import org.apache.ignite.internal.failure.FailureType;
import org.apache.ignite.internal.hlc.ClockService;
import org.apache.ignite.internal.hlc.HybridTimestamp;
import org.apache.ignite.internal.index.IndexBuilder;
import org.apache.ignite.internal.index.IndexManagementUtils;
import org.apache.ignite.internal.index.IndexManager;
import org.apache.ignite.internal.lang.NodeStoppingException;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.network.ClusterService;
import org.apache.ignite.internal.network.InternalClusterNode;
import org.apache.ignite.internal.partition.replicator.PartitionReplicaLifecycleManager;
import org.apache.ignite.internal.partition.replicator.TableTxRwOperationTracker;
import org.apache.ignite.internal.partition.replicator.ZonePartitionReplicaListener;
import org.apache.ignite.internal.partition.replicator.ZoneResourcesManager;
import org.apache.ignite.internal.placementdriver.PlacementDriver;
import org.apache.ignite.internal.placementdriver.PrimaryReplicaAwaitTimeoutException;
import org.apache.ignite.internal.placementdriver.ReplicaMeta;
import org.apache.ignite.internal.placementdriver.event.PrimaryReplicaEvent;
import org.apache.ignite.internal.placementdriver.event.PrimaryReplicaEventParameters;
import org.apache.ignite.internal.replicator.ReplicationGroupId;
import org.apache.ignite.internal.replicator.ZonePartitionId;
import org.apache.ignite.internal.storage.MvPartitionStorage;
import org.apache.ignite.internal.storage.StorageClosedException;
import org.apache.ignite.internal.storage.engine.MvTableStorage;
import org.apache.ignite.internal.storage.index.IndexStorage;
import org.apache.ignite.internal.util.CompletableFutures;
import org.apache.ignite.internal.util.ExceptionUtils;
import org.apache.ignite.internal.util.IgniteBusyLock;
import org.apache.ignite.internal.util.IgniteSpinBusyLock;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.internal.util.PendingComparableValuesTracker;
import org.jetbrains.annotations.Nullable;

class IndexBuildController
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(IndexBuildController.class);
    private final IndexBuilder indexBuilder;
    private final IndexManager indexManager;
    private final CatalogService catalogService;
    private final ClusterService clusterService;
    private final PlacementDriver placementDriver;
    private final ClockService clockService;
    private final PartitionReplicaLifecycleManager partitionReplicaLifecycleManager;
    private final FailureProcessor failureProcessor;
    private final IgniteSpinBusyLock busyLock = new IgniteSpinBusyLock();
    private final AtomicBoolean closeGuard = new AtomicBoolean();
    private final Set<ZonePartitionId> primaryReplicaIds = ConcurrentHashMap.newKeySet();

    IndexBuildController(IndexBuilder indexBuilder, IndexManager indexManager, CatalogService catalogService, ClusterService clusterService, PlacementDriver placementDriver, ClockService clockService, PartitionReplicaLifecycleManager partitionReplicaLifecycleManager, FailureProcessor failureProcessor) {
        this.indexBuilder = indexBuilder;
        this.indexManager = indexManager;
        this.catalogService = catalogService;
        this.clusterService = clusterService;
        this.placementDriver = placementDriver;
        this.clockService = clockService;
        this.partitionReplicaLifecycleManager = partitionReplicaLifecycleManager;
        this.failureProcessor = failureProcessor;
    }

    public void start() {
        IgniteUtils.inBusyLock((IgniteBusyLock)this.busyLock, this::addListeners);
    }

    public void close() {
        if (!this.closeGuard.compareAndSet(false, true)) {
            return;
        }
        this.busyLock.block();
        this.indexBuilder.close();
    }

    private void addListeners() {
        this.catalogService.listen((Event)CatalogEvent.INDEX_BUILDING, EventListener.fromConsumer(this::onIndexBuilding));
        this.catalogService.listen((Event)CatalogEvent.INDEX_REMOVED, EventListener.fromConsumer(this::onIndexRemoved));
        this.placementDriver.listen((Event)PrimaryReplicaEvent.PRIMARY_REPLICA_ELECTED, EventListener.fromConsumer(this::onPrimaryReplicaElected));
    }

    private void onIndexBuilding(StartBuildingIndexEventParameters parameters) {
        IgniteUtils.inBusyLockAsync((IgniteBusyLock)this.busyLock, () -> {
            Catalog catalog = this.catalogService.catalog(parameters.catalogVersion());
            assert (catalog != null) : "Failed to find a catalog for the specified version [version=" + parameters.catalogVersion() + ", earliestVersion=" + this.catalogService.earliestCatalogVersion() + ", latestVersion=" + this.catalogService.latestCatalogVersion() + "].";
            CatalogIndexDescriptor indexDescriptor = catalog.index(parameters.indexId());
            assert (indexDescriptor != null) : "Failed to find an index descriptor for the specified index [indexId=" + parameters.indexId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            CatalogTableDescriptor tableDescriptor = catalog.table(indexDescriptor.tableId());
            assert (tableDescriptor != null) : "Failed to find a table descriptor for the specified index [indexId=" + parameters.indexId() + ", tableId=" + indexDescriptor.tableId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            CatalogZoneDescriptor zoneDescriptor = catalog.zone(tableDescriptor.zoneId());
            assert (zoneDescriptor != null) : "Failed to find a zone descriptor for the specified table [indexId=" + parameters.indexId() + ", tableId=" + indexDescriptor.tableId() + ", catalogVersion=" + parameters.catalogVersion() + "].";
            ArrayList<CompletionStage> startBuildIndexFutures = new ArrayList<CompletionStage>();
            for (ZonePartitionId zoneReplicaId : this.primaryReplicaIds) {
                int zoneId = zoneReplicaId.zoneId();
                int partitionId = zoneReplicaId.partitionId();
                boolean needToProcessPartition = zoneReplicaId.zoneId() == zoneDescriptor.id();
                if (!needToProcessPartition) continue;
                CompletionStage startBuildIndexFuture = this.indexManager.getMvTableStorage(parameters.causalityToken(), indexDescriptor.tableId()).thenCompose(mvTableStorage -> {
                    HybridTimestamp buildAttemptTimestamp = this.clockService.now();
                    return this.awaitPrimaryReplica(zoneReplicaId, buildAttemptTimestamp).thenAccept(replicaMeta -> this.tryScheduleBuildIndex(zoneId, indexDescriptor.tableId(), partitionId, zoneReplicaId, indexDescriptor, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta, buildAttemptTimestamp));
                });
                startBuildIndexFutures.add(startBuildIndexFuture);
            }
            return CompletableFutures.allOf(startBuildIndexFutures);
        }).whenComplete((res, ex) -> {
            if (ex != null && !ExceptionUtils.hasCause((Throwable)ex, (Class[])new Class[]{NodeStoppingException.class, StorageClosedException.class})) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, ex));
            }
        });
    }

    private void onIndexRemoved(RemoveIndexEventParameters parameters) {
        IgniteUtils.inBusyLockAsync((IgniteBusyLock)this.busyLock, () -> {
            this.indexBuilder.stopBuildingIndexes(parameters.indexId());
            return CompletableFutures.nullCompletedFuture();
        }).whenComplete((res, ex) -> {
            if (ex != null) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, ex));
            }
        });
    }

    private void onPrimaryReplicaElected(PrimaryReplicaEventParameters parameters) {
        IgniteUtils.inBusyLockAsync((IgniteBusyLock)this.busyLock, () -> {
            assert (parameters.groupId() instanceof ZonePartitionId) : "Primary replica ID must be of type ZonePartitionId [groupId=" + parameters.groupId() + ", class=" + parameters.groupId().getClass().getSimpleName() + "].";
            ZonePartitionId primaryReplicaId = (ZonePartitionId)parameters.groupId();
            if (IndexManagementUtils.isLocalNode(this.clusterService, parameters.leaseholderId())) {
                this.primaryReplicaIds.add(primaryReplicaId);
                Catalog catalog = this.catalogService.catalog(this.catalogService.latestCatalogVersion());
                CatalogZoneDescriptor zoneDescriptor = catalog.zone(primaryReplicaId.zoneId());
                if (zoneDescriptor == null) {
                    return CompletableFutures.nullCompletedFuture();
                }
                ArrayList<CompletionStage> indexFutures = new ArrayList<CompletionStage>();
                for (CatalogTableDescriptor tableDescriptor : catalog.tables(zoneDescriptor.id())) {
                    CompletionStage future = this.indexManager.getMvTableStorage(parameters.causalityToken(), tableDescriptor.id()).thenCompose(mvTableStorage -> {
                        HybridTimestamp buildAttemptTimestamp = this.clockService.now();
                        return this.awaitPrimaryReplica(primaryReplicaId, buildAttemptTimestamp).thenAccept(replicaMeta -> this.tryScheduleBuildIndexesForNewPrimaryReplica(catalog, tableDescriptor, primaryReplicaId, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta, buildAttemptTimestamp));
                    });
                    indexFutures.add(future);
                }
                return CompletableFutures.allOf(indexFutures);
            }
            this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
            return CompletableFutures.nullCompletedFuture();
        }).whenComplete((res, ex) -> {
            if (ex != null && !ExceptionUtils.hasCause((Throwable)ex, (Class[])new Class[]{NodeStoppingException.class, StorageClosedException.class})) {
                this.failureProcessor.process(new FailureContext(FailureType.CRITICAL_ERROR, ex));
            }
        });
    }

    private void tryScheduleBuildIndexesForNewPrimaryReplica(Catalog catalog, CatalogTableDescriptor tableDescriptor, ZonePartitionId primaryReplicaId, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        IgniteUtils.inBusyLock((IgniteBusyLock)this.busyLock, () -> {
            if (this.isLeaseExpired(replicaMeta, buildAttemptTimestamp)) {
                LOG.info("Lease has expired (on new primary), stopping build index process [groupId={}, localNode={}, primaryReplica={}.", new Object[]{primaryReplicaId, this.localNode(), replicaMeta});
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            for (CatalogIndexDescriptor indexDescriptor : catalog.indexes(tableDescriptor.id())) {
                if (indexDescriptor.status() == CatalogIndexStatus.BUILDING) {
                    this.scheduleBuildIndex(tableDescriptor.zoneId(), tableDescriptor.id(), primaryReplicaId.partitionId(), indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta), this.clockService.current());
                    continue;
                }
                if (indexDescriptor.status() != CatalogIndexStatus.AVAILABLE) continue;
                this.scheduleBuildIndexAfterDisasterRecovery(tableDescriptor.zoneId(), tableDescriptor.id(), primaryReplicaId.partitionId(), indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta));
            }
        });
    }

    private void tryScheduleBuildIndex(int zoneId, int tableId, int partitionId, ZonePartitionId primaryReplicaId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        assert (primaryReplicaId.zoneId() == zoneId && primaryReplicaId.partitionId() == partitionId) : "Primary replica identifier mismatched [zoneId=" + zoneId + ", tableId=" + tableId + ", partitionId=" + partitionId + ", primaryReplicaId=" + primaryReplicaId + ", primaryReplicaCls=" + primaryReplicaId.getClass().getSimpleName() + "].";
        IgniteUtils.inBusyLock((IgniteBusyLock)this.busyLock, () -> {
            if (this.isLeaseExpired(replicaMeta, buildAttemptTimestamp)) {
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            this.scheduleBuildIndex(zoneId, tableId, partitionId, indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta), this.clockService.current());
        });
    }

    private void stopBuildingIndexesIfPrimaryExpired(ZonePartitionId replicaId) {
        if (this.primaryReplicaIds.remove(replicaId)) {
            this.indexBuilder.stopBuildingZoneIndexes(replicaId.zoneId(), replicaId.partitionId());
        }
    }

    private CompletableFuture<ReplicaMeta> awaitPrimaryReplica(ZonePartitionId replicaId, HybridTimestamp timestamp) {
        return ((CompletableFuture)this.placementDriver.awaitPrimaryReplica((ReplicationGroupId)replicaId, timestamp, 10L, TimeUnit.SECONDS).handle((replicaMeta, throwable) -> {
            if (throwable != null) {
                Throwable unwrapThrowable = ExceptionUtils.unwrapCause((Throwable)throwable);
                if (unwrapThrowable instanceof PrimaryReplicaAwaitTimeoutException) {
                    return this.awaitPrimaryReplica(replicaId, timestamp);
                }
                return CompletableFuture.failedFuture(unwrapThrowable);
            }
            return CompletableFuture.completedFuture(replicaMeta);
        })).thenCompose(Function.identity());
    }

    private void scheduleBuildIndex(int zoneId, int tableId, int partitionId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken, HybridTimestamp initialOperationTimestamp) {
        ZonePartitionId zonePartitionId = new ZonePartitionId(zoneId, partitionId);
        ZoneResourcesManager.ZonePartitionResources resources = this.partitionReplicaLifecycleManager.zonePartitionResourcesOrNull(zonePartitionId);
        if (resources == null) {
            return;
        }
        TableTxRwOperationTracker txRwOperationTracker = IndexBuildController.txRwOperationTracker(zonePartitionId, tableId, resources, indexDescriptor);
        if (txRwOperationTracker == null) {
            return;
        }
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, zoneId, tableId, partitionId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, partitionId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndex(zoneId, tableId, partitionId, indexDescriptor.id(), indexStorage, mvPartition, txRwOperationTracker, (PendingComparableValuesTracker<HybridTimestamp, Void>)resources.safeTimeTracker(), this.localNode(), enlistmentConsistencyToken, initialOperationTimestamp);
    }

    private void scheduleBuildIndexAfterDisasterRecovery(int zoneId, int tableId, int partitionId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken) {
        ZonePartitionId zonePartitionId = new ZonePartitionId(zoneId, partitionId);
        ZoneResourcesManager.ZonePartitionResources resources = this.partitionReplicaLifecycleManager.zonePartitionResourcesOrNull(zonePartitionId);
        if (resources == null) {
            return;
        }
        TableTxRwOperationTracker txRwOperationTracker = IndexBuildController.txRwOperationTracker(zonePartitionId, tableId, resources, indexDescriptor);
        if (txRwOperationTracker == null) {
            return;
        }
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, zoneId, tableId, partitionId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, partitionId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndexAfterDisasterRecovery(zoneId, tableId, partitionId, indexDescriptor.id(), indexStorage, mvPartition, txRwOperationTracker, (PendingComparableValuesTracker<HybridTimestamp, Void>)resources.safeTimeTracker(), this.localNode(), enlistmentConsistencyToken, this.clockService.current());
    }

    @Nullable
    private static TableTxRwOperationTracker txRwOperationTracker(ZonePartitionId zonePartitionId, int tableId, ZoneResourcesManager.ZonePartitionResources resources, CatalogIndexDescriptor indexDescriptor) {
        CompletableFuture replicaListenerFuture = resources.replicaListenerFuture();
        assert (replicaListenerFuture.isDone()) : "Replica listener future is not done for [zonePartitionId=" + zonePartitionId + "].";
        ZonePartitionReplicaListener replicaListener = (ZonePartitionReplicaListener)replicaListenerFuture.join();
        @Nullable TableTxRwOperationTracker txRwOperationTracker = replicaListener.txRwOperationTracker(tableId);
        if (txRwOperationTracker == null) {
            LOG.info("Tracker is null, skipping index build scheduling [zoneId={}, tableId={}, partitionId={}, indexId={}]", new Object[]{zonePartitionId.zoneId(), tableId, zonePartitionId.partitionId(), indexDescriptor.id()});
        }
        return txRwOperationTracker;
    }

    private InternalClusterNode localNode() {
        return IndexManagementUtils.localNode(this.clusterService);
    }

    private boolean isLeaseExpired(ReplicaMeta replicaMeta, HybridTimestamp buildAttemptTimestamp) {
        return !IndexManagementUtils.isPrimaryReplica(replicaMeta, this.localNode(), buildAttemptTimestamp);
    }

    private static long enlistmentConsistencyToken(ReplicaMeta replicaMeta) {
        return replicaMeta.getStartTime().longValue();
    }

    private static MvPartitionStorage mvPartitionStorage(MvTableStorage mvTableStorage, int zoneId, int tableId, int partitionId) {
        MvPartitionStorage mvPartition = mvTableStorage.getMvPartition(partitionId);
        assert (mvPartition != null) : "Partition storage is missing [zoneId=" + zoneId + ", tableId=" + tableId + ", partitionId=" + partitionId + "].";
        return mvPartition;
    }

    private static IndexStorage indexStorage(MvTableStorage mvTableStorage, int partitionId, CatalogIndexDescriptor indexDescriptor) {
        IndexStorage indexStorage = mvTableStorage.getIndex(partitionId, indexDescriptor.id());
        assert (indexStorage != null) : "Index storage is missing [partitionId=" + partitionId + ", indexId=" + indexDescriptor.id() + "].";
        return indexStorage;
    }
}

