/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.commandline.dr.subcommands;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import org.apache.ignite.internal.client.GridClient;
import org.apache.ignite.internal.client.GridClientCompute;
import org.apache.ignite.internal.client.GridClientConfiguration;
import org.apache.ignite.internal.client.GridClientDisconnectedException;
import org.apache.ignite.internal.client.GridClientException;
import org.apache.ignite.internal.client.GridClientNode;
import org.apache.ignite.internal.commandline.CommandArgIterator;
import org.apache.ignite.internal.commandline.argument.CommandArg;
import org.apache.ignite.internal.commandline.argument.CommandArgUtils;
import org.apache.ignite.internal.commandline.dr.DrSubCommandsList;
import org.apache.ignite.internal.util.typedef.F;
import org.apache.ignite.internal.util.typedef.T2;
import org.apache.ignite.internal.visor.VisorTaskArgument;
import org.apache.ignite.internal.visor.dr.VisorDrCacheTaskArgs;
import org.apache.ignite.internal.visor.dr.VisorDrCacheTaskResult;

import static org.apache.ignite.internal.commandline.CommandHandler.DELIM;
import static org.apache.ignite.internal.commandline.CommandLogger.INDENT;
import static org.apache.ignite.internal.commandline.dr.subcommands.DrFSTCommand.ParseStart.SYNC_MODE;

/** */
public class DrCacheCommand extends
    DrAbstractRemoteSubCommand<VisorDrCacheTaskArgs, VisorDrCacheTaskResult, DrCacheCommand.DrCacheArguments>
{
    /** Config parameter. */
    public static final String CONFIG_PARAM = "--config";

    /** Metrics parameter. */
    public static final String METRICS_PARAM = "--metrics";

    /** Cache filter parameter. */
    public static final String CACHE_FILTER_PARAM = "--cache-filter";

    /** Sender group parameter. */
    public static final String SENDER_GROUP_PARAM = "--sender-group";

    /** Action parameter. */
    public static final String ACTION_PARAM = "--action";

    /** {@inheritDoc} */
    @Override protected String visorTaskName() {
        throw new UnsupportedOperationException("visorTaskName");
    }

    /** {@inheritDoc} */
    @Override public DrCacheArguments parseArguments0(CommandArgIterator argIter) {
        String regex = argIter.nextArg("Cache name regex expected.");

        if (CommandArgIterator.isCommandOrOption(regex))
            throw new IllegalArgumentException("Cache name regex expected.");

        Pattern pattern;

        try {
            pattern = Pattern.compile(regex);
        }
        catch (PatternSyntaxException e) {
            throw new IllegalArgumentException("Cache name regex is not valid.", e);
        }

        boolean cfg = false;
        boolean metrics = false;
        CacheFilter cacheFilter = CacheFilter.ALL;
        SenderGroup sndGrp = SenderGroup.ALL;
        String sndGrpName = null;
        Action act = null;
        boolean fstSyncMode = false;

        String nextArg;

        //noinspection LabeledStatement
        args_loop: while ((nextArg = argIter.peekNextArg()) != null) {
            switch (nextArg.toLowerCase(Locale.ENGLISH)) {
                case CONFIG_PARAM:
                    argIter.nextArg(null);
                    cfg = true;

                    break;

                case METRICS_PARAM:
                    argIter.nextArg(null);
                    metrics = true;

                    break;

                case CACHE_FILTER_PARAM: {
                    argIter.nextArg(null);

                    cacheFilter = argIter.nextEnumOrFail(CacheFilter.class, CACHE_FILTER_PARAM);

                    break;
                }

                case SENDER_GROUP_PARAM: {
                    argIter.nextArg(null);

                    String arg = argIter.nextArgValue(SENDER_GROUP_PARAM);

                    sndGrp = CommandArgUtils.ofEnum(SenderGroup.class, arg);

                    if (sndGrp == null)
                        sndGrpName = arg;

                    break;
                }

                case ACTION_PARAM: {
                    argIter.nextArg(null);

                    act = argIter.nextCmdArgOrFail(Action.class, ACTION_PARAM);

                    break;
                }

                case SYNC_MODE: {
                    argIter.nextArg(null);

                    fstSyncMode = true;
                }

                default:
                    //noinspection BreakStatementWithLabel
                    break args_loop;
            }
        }

        return new DrCacheArguments(regex, pattern, cfg, metrics, cacheFilter, sndGrp, sndGrpName, act, (byte)0, fstSyncMode);
    }

    /** {@inheritDoc} */
    @Override public String confirmationPrompt() {
        if (arg().action != null)
            return "Warning: this command will change data center replication state for selected caches.";

        return null;
    }

    /** {@inheritDoc} */
    @Override protected VisorDrCacheTaskResult execute0(GridClientConfiguration clientCfg, GridClient client) throws Exception {
        return execute0(client, arg());
    }

    /** */
    public static VisorDrCacheTaskResult execute0(
        GridClient client,
        DrCacheArguments arg
    ) throws GridClientException {
        GridClientCompute compute = client.compute();

        Collection<GridClientNode> nodes = compute.nodes();

        Pattern cacheNamePattern = arg.pattern;

        List<UUID> nodeIds = nodes.stream()
            .filter(DrAbstractRemoteSubCommand::drControlUtilitySupported)
            .map(GridClientNode::nodeId)
            .collect(Collectors.toList());

        if (F.isEmpty(nodeIds))
            throw new GridClientDisconnectedException("Connectable nodes not found", null);

        if (arg.remoteDataCenterId == 0 && arg.action != null) {
            Map<String, UUID> cacheNameToNodeMap = new HashMap<>();

            for (GridClientNode node : nodes) {
                for (String cacheName : node.caches().keySet()) {
                    if (cacheNamePattern.matcher(cacheName).matches())
                        cacheNameToNodeMap.putIfAbsent(cacheName, node.nodeId());
                }
            }

            arg.cacheNamesMap = cacheNameToNodeMap;
        }
        else if (arg.remoteDataCenterId != 0) {
            for (GridClientNode node : nodes) {
                if (node.attribute("plugins.gg.replication.ist.snd.hub") != null ||
                    node.attribute("plugins.gg.replication.snd.hub") != null) {
                    arg.actionCoordinator = node.nodeId();

                    break;
                }
            }
        }

        return compute.projection(DrAbstractRemoteSubCommand::drControlUtilitySupported).execute(
            "org.gridgain.grid.internal.visor.dr.console.VisorDrCacheTask",
            new VisorTaskArgument<>(nodeIds, arg.toVisorArgs(), false)
        );
    }

    /** {@inheritDoc} */
    @Override protected void printResult(VisorDrCacheTaskResult res, Logger log) {
        printUnrecognizedNodesMessage(log, false);

        log.info("Data Center ID: " + res.getDataCenterId());

        log.info(DELIM);

        if (res.getDataCenterId() == 0) {
            log.info("Data Replication state: is not configured.");

            return;
        }

        List<String> cacheNames = res.getCacheNames();
        if (cacheNames.isEmpty()) {
            log.info("No matching caches found");

            return;
        }

        log.info(String.format("%d matching cache(s): %s", cacheNames.size(), cacheNames));

        for (String cacheName : cacheNames) {
            List<T2<String, Object>> cacheSndCfg = res.getSenderConfig().get(cacheName);

            printList(log, cacheSndCfg, String.format(
                "Sender configuration for cache \"%s\":",
                cacheName
            ));

            List<T2<String, Object>> cacheRcvCfg = res.getReceiverConfig().get(cacheName);

            printList(log, cacheRcvCfg, String.format(
                "Receiver configuration for cache \"%s\":",
                cacheName
            ));
        }

        for (String cacheName : cacheNames) {
            List<T2<String, Object>> cacheSndMetrics = res.getSenderMetrics().get(cacheName);

            printList(log, cacheSndMetrics, String.format(
                "Sender metrics for cache \"%s\":",
                cacheName
            ));

            List<T2<String, Object>> cacheRcvMetrics = res.getReceiverMetrics().get(cacheName);

            printList(log, cacheRcvMetrics, String.format(
                "Receiver metrics for cache \"%s\":",
                cacheName
            ));
        }

        for (String msg : res.getResultMessages())
            log.info(msg);
    }

    /** */
    private static void printList(Logger log, List<T2<String, Object>> cfg, String s) {
        if (cfg != null && !cfg.isEmpty()) {
            log.info(s);

            for (T2<String, Object> t2 : cfg)
                log.info(String.format(INDENT + "%s=%s", t2.toArray()));
        }
    }

    /** {@inheritDoc} */
    @Override public String name() {
        return DrSubCommandsList.CACHE.text();
    }

    /** */
    @SuppressWarnings("PublicInnerClass") public enum CacheFilter {
        /** All. */ ALL,
        /** Sending. */ SENDING,
        /** Receiving. */ RECEIVING,
        /** Paused. */ PAUSED,
        /** Error. */ ERROR
    }

    /** */
    @SuppressWarnings("PublicInnerClass") public enum SenderGroup {
        /** All. */ ALL,
        /** Default. */ DEFAULT,
        /** None. */ NONE
    }

    /** */
    @SuppressWarnings("PublicInnerClass") public enum Action implements CommandArg {
        /** Stop. */ STOP("stop"),

        /** Start. */ START("start"),

        /** Full state transfer. */
        FULL_STATE_TRANSFER("full-state-transfer");

        /** String representation. */
        private final String name;

        /** */
        Action(String name) {
            this.name = name;
        }

        @Override public String argName() {
            return name;
        }

        /** {@inheritDoc} */
        @Override public String toString() {
            return name;
        }
    }

    /** */
    @SuppressWarnings("PublicInnerClass")
    public static class DrCacheArguments implements DrAbstractRemoteSubCommand.Arguments<VisorDrCacheTaskArgs> {
        /** Regex. */
        private final String regex;

        /** Pattern. */
        private final Pattern pattern;

        /** Config. */
        private final boolean config;

        /** Metrics. */
        private final boolean metrics;

        /** Filter. */
        private final CacheFilter filter;

        /** Sender group. */
        private final SenderGroup senderGroup;

        /** Sender group name. */
        private final String senderGroupName;

        /** Action. */
        private final Action action;

        /** Remote data center id. */
        private final byte remoteDataCenterId;

        /** Cache names map. */
        private Map<String, UUID> cacheNamesMap;

        /** Action coordinator. */
        private UUID actionCoordinator;

        /** FST sync mode. */
        private boolean fstSyncMode;

        /** */
        public DrCacheArguments(
            String regex,
            Pattern pattern,
            boolean config,
            boolean metrics,
            CacheFilter filter,
            SenderGroup senderGroup,
            String senderGroupName,
            Action action,
            byte remoteDataCenterId,
            boolean fstSyncMode
        ) {
            this.regex = regex;
            this.pattern = pattern;
            this.config = config;
            this.metrics = metrics;
            this.filter = filter;
            this.senderGroup = senderGroup;
            this.senderGroupName = senderGroupName;
            this.action = action;
            this.remoteDataCenterId = remoteDataCenterId;
            this.fstSyncMode = fstSyncMode;
        }

        /** */
        public UUID getActionCoordinator() {
            return actionCoordinator;
        }

        /** {@inheritDoc} */
        @Override public VisorDrCacheTaskArgs toVisorArgs() {
            return new VisorDrCacheTaskArgs(
                regex,
                config,
                metrics,
                filter.ordinal(),
                senderGroup == null ? VisorDrCacheTaskArgs.SENDER_GROUP_NAMED : senderGroup.ordinal(),
                senderGroupName,
                action == null ? VisorDrCacheTaskArgs.ACTION_NONE : action.ordinal(),
                remoteDataCenterId,
                cacheNamesMap,
                actionCoordinator,
                fstSyncMode
            );
        }
    }
}
