/*
 * Decompiled with CFR 0.152.
 */
package org.gridgain.internal.h2.store;

import java.io.IOException;
import java.io.OutputStream;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Properties;
import org.gridgain.internal.h2.Driver;
import org.gridgain.internal.h2.message.DbException;
import org.gridgain.internal.h2.message.Trace;
import org.gridgain.internal.h2.message.TraceSystem;
import org.gridgain.internal.h2.store.FileLockMethod;
import org.gridgain.internal.h2.store.fs.FileUtils;
import org.gridgain.internal.h2.util.MathUtils;
import org.gridgain.internal.h2.util.NetUtils;
import org.gridgain.internal.h2.util.SortedProperties;
import org.gridgain.internal.h2.util.StringUtils;
import org.gridgain.internal.h2.value.Transfer;

public class FileLock
implements Runnable {
    private static final String MAGIC = "FileLock";
    private static final String FILE = "file";
    private static final String SOCKET = "socket";
    private static final String SERIALIZED = "serialized";
    private static final int RANDOM_BYTES = 16;
    private static final int SLEEP_GAP = 25;
    private static final int TIME_GRANULARITY = 2000;
    private volatile String fileName;
    private volatile ServerSocket serverSocket;
    private volatile boolean locked;
    private final int sleep;
    private final Trace trace;
    private long lastWrite;
    private String method;
    private Properties properties;
    private String uniqueId;
    private Thread watchdog;

    public FileLock(TraceSystem traceSystem, String fileName, int sleep) {
        this.trace = traceSystem == null ? null : traceSystem.getTrace(4);
        this.fileName = fileName;
        this.sleep = sleep;
    }

    public synchronized void lock(FileLockMethod fileLockMethod) {
        this.checkServer();
        if (this.locked) {
            DbException.throwInternalError("already locked");
        }
        switch (fileLockMethod) {
            case FILE: {
                this.lockFile();
                break;
            }
            case SOCKET: {
                this.lockSocket();
                break;
            }
            case SERIALIZED: {
                this.lockSerialized();
                break;
            }
        }
        this.locked = true;
    }

    public synchronized void unlock() {
        if (!this.locked) {
            return;
        }
        this.locked = false;
        try {
            if (this.watchdog != null) {
                this.watchdog.interrupt();
            }
        }
        catch (Exception e) {
            this.trace.debug(e, "unlock");
        }
        try {
            if (this.fileName != null && this.load().equals(this.properties)) {
                FileUtils.delete(this.fileName);
            }
            if (this.serverSocket != null) {
                this.serverSocket.close();
            }
        }
        catch (Exception e) {
            this.trace.debug(e, "unlock");
        }
        finally {
            this.fileName = null;
            this.serverSocket = null;
        }
        try {
            if (this.watchdog != null) {
                this.watchdog.join();
            }
        }
        catch (Exception e) {
            this.trace.debug(e, "unlock");
        }
        finally {
            this.watchdog = null;
        }
    }

    public void setProperty(String key, String value) {
        if (value == null) {
            this.properties.remove(key);
        } else {
            this.properties.put(key, value);
        }
    }

    public Properties save() {
        try {
            try (OutputStream out = FileUtils.newOutputStream(this.fileName, false);){
                this.properties.store(out, MAGIC);
            }
            this.lastWrite = FileUtils.lastModified(this.fileName);
            if (this.trace.isDebugEnabled()) {
                this.trace.debug("save " + this.properties);
            }
            return this.properties;
        }
        catch (IOException e) {
            throw FileLock.getExceptionFatal("Could not save properties " + this.fileName, e);
        }
    }

    private void checkServer() {
        Properties prop = this.load();
        String server = prop.getProperty("server");
        if (server == null) {
            return;
        }
        boolean running = false;
        String id = prop.getProperty("id");
        try {
            Socket socket = NetUtils.createSocket(server, 9092, false);
            Transfer transfer = new Transfer(null, socket);
            transfer.init();
            transfer.writeInt(9);
            transfer.writeInt(18);
            transfer.writeString(null);
            transfer.writeString(null);
            transfer.writeString(id);
            transfer.writeInt(14);
            transfer.flush();
            int state = transfer.readInt();
            if (state == 1) {
                running = true;
            }
            transfer.close();
            socket.close();
        }
        catch (IOException e) {
            return;
        }
        if (running) {
            DbException e = DbException.get(90020, "Server is running");
            throw e.addSQL(server + "/" + id);
        }
    }

    public Properties load() {
        IOException lastException = null;
        for (int i = 0; i < 5; ++i) {
            try {
                SortedProperties p2 = SortedProperties.loadProperties(this.fileName);
                if (this.trace.isDebugEnabled()) {
                    this.trace.debug("load " + p2);
                }
                return p2;
            }
            catch (IOException e) {
                lastException = e;
                continue;
            }
        }
        throw FileLock.getExceptionFatal("Could not load properties " + this.fileName, lastException);
    }

    private void waitUntilOld() {
        for (int i = 0; i < 160; ++i) {
            long last = FileUtils.lastModified(this.fileName);
            long dist = System.currentTimeMillis() - last;
            if (dist < -2000L) {
                try {
                    Thread.sleep(2L * (long)this.sleep);
                }
                catch (Exception e) {
                    this.trace.debug(e, "sleep");
                }
                return;
            }
            if (dist > 2000L) {
                return;
            }
            try {
                Thread.sleep(25L);
                continue;
            }
            catch (Exception e) {
                this.trace.debug(e, "sleep");
            }
        }
        throw FileLock.getExceptionFatal("Lock file recently modified", null);
    }

    private void setUniqueId() {
        byte[] bytes = MathUtils.secureRandomBytes(16);
        String random = StringUtils.convertBytesToHex(bytes);
        this.uniqueId = Long.toHexString(System.currentTimeMillis()) + random;
        this.properties.setProperty("id", this.uniqueId);
    }

    private void lockSerialized() {
        this.method = SERIALIZED;
        FileUtils.createDirectories(FileUtils.getParent(this.fileName));
        if (!FileUtils.createFile(this.fileName)) {
            try {
                this.properties = this.load();
            }
            catch (DbException dbException) {
                // empty catch block
            }
            return;
        }
        this.properties = new SortedProperties();
        this.properties.setProperty("method", String.valueOf(this.method));
        this.setUniqueId();
        this.save();
    }

    private void lockFile() {
        this.method = FILE;
        this.properties = new SortedProperties();
        this.properties.setProperty("method", String.valueOf(this.method));
        this.setUniqueId();
        FileUtils.createDirectories(FileUtils.getParent(this.fileName));
        if (!FileUtils.createFile(this.fileName)) {
            this.waitUntilOld();
            String m22 = this.load().getProperty("method", FILE);
            if (!m22.equals(FILE)) {
                throw FileLock.getExceptionFatal("Unsupported lock method " + m22, null);
            }
            this.save();
            FileLock.sleep(2 * this.sleep);
            if (!this.load().equals(this.properties)) {
                throw this.getExceptionAlreadyInUse("Locked by another process: " + this.fileName);
            }
            FileUtils.delete(this.fileName);
            if (!FileUtils.createFile(this.fileName)) {
                throw FileLock.getExceptionFatal("Another process was faster", null);
            }
        }
        this.save();
        FileLock.sleep(25);
        if (!this.load().equals(this.properties)) {
            this.fileName = null;
            throw FileLock.getExceptionFatal("Concurrent update", null);
        }
        this.locked = true;
        this.watchdog = new Thread((Runnable)this, "H2 File Lock Watchdog " + this.fileName);
        Driver.setThreadContextClassLoader(this.watchdog);
        this.watchdog.setDaemon(true);
        this.watchdog.setPriority(9);
        this.watchdog.start();
    }

    private void lockSocket() {
        this.method = SOCKET;
        this.properties = new SortedProperties();
        this.properties.setProperty("method", String.valueOf(this.method));
        this.setUniqueId();
        String ipAddress = NetUtils.getLocalAddress();
        FileUtils.createDirectories(FileUtils.getParent(this.fileName));
        if (!FileUtils.createFile(this.fileName)) {
            InetAddress address;
            this.waitUntilOld();
            long read = FileUtils.lastModified(this.fileName);
            Properties p2 = this.load();
            String m22 = p2.getProperty("method", SOCKET);
            if (m22.equals(FILE)) {
                this.lockFile();
                return;
            }
            if (!m22.equals(SOCKET)) {
                throw FileLock.getExceptionFatal("Unsupported lock method " + m22, null);
            }
            String ip = p2.getProperty("ipAddress", ipAddress);
            if (!ipAddress.equals(ip)) {
                throw this.getExceptionAlreadyInUse("Locked by another computer: " + ip);
            }
            String port = p2.getProperty("port", "0");
            int portId = Integer.parseInt(port);
            try {
                address = InetAddress.getByName(ip);
            }
            catch (UnknownHostException e) {
                throw FileLock.getExceptionFatal("Unknown host " + ip, e);
            }
            for (int i = 0; i < 3; ++i) {
                try {
                    Socket s2 = new Socket(address, portId);
                    s2.close();
                    throw this.getExceptionAlreadyInUse("Locked by another process");
                }
                catch (BindException e) {
                    throw FileLock.getExceptionFatal("Bind Exception", null);
                }
                catch (ConnectException e) {
                    this.trace.debug(e, "socket not connected to port " + port);
                    continue;
                }
                catch (IOException e) {
                    throw FileLock.getExceptionFatal("IOException", null);
                }
            }
            if (read != FileUtils.lastModified(this.fileName)) {
                throw FileLock.getExceptionFatal("Concurrent update", null);
            }
            FileUtils.delete(this.fileName);
            if (!FileUtils.createFile(this.fileName)) {
                throw FileLock.getExceptionFatal("Another process was faster", null);
            }
        }
        try {
            this.serverSocket = NetUtils.createServerSocket(0, false);
            int port = this.serverSocket.getLocalPort();
            this.properties.setProperty("ipAddress", ipAddress);
            this.properties.setProperty("port", Integer.toString(port));
        }
        catch (Exception e) {
            this.trace.debug(e, "lock");
            this.serverSocket = null;
            this.lockFile();
            return;
        }
        this.save();
        this.locked = true;
        this.watchdog = new Thread((Runnable)this, "H2 File Lock Watchdog (Socket) " + this.fileName);
        this.watchdog.setDaemon(true);
        this.watchdog.start();
    }

    private static void sleep(int time) {
        try {
            Thread.sleep(time);
        }
        catch (InterruptedException e) {
            throw FileLock.getExceptionFatal("Sleep interrupted", e);
        }
    }

    private static DbException getExceptionFatal(String reason, Throwable t2) {
        return DbException.get(8000, t2, reason);
    }

    private DbException getExceptionAlreadyInUse(String reason) {
        DbException e = DbException.get(90020, reason);
        if (this.fileName != null) {
            try {
                Properties prop = this.load();
                String server = prop.getProperty("server");
                if (server != null) {
                    String serverId = server + "/" + prop.getProperty("id");
                    e = e.addSQL(serverId);
                }
            }
            catch (DbException dbException) {
                // empty catch block
            }
        }
        return e;
    }

    public static FileLockMethod getFileLockMethod(String method) {
        if (method == null || method.equalsIgnoreCase("FILE")) {
            return FileLockMethod.FILE;
        }
        if (method.equalsIgnoreCase("NO")) {
            return FileLockMethod.NO;
        }
        if (method.equalsIgnoreCase("SOCKET")) {
            return FileLockMethod.SOCKET;
        }
        if (method.equalsIgnoreCase("SERIALIZED")) {
            return FileLockMethod.SERIALIZED;
        }
        if (method.equalsIgnoreCase("FS")) {
            return FileLockMethod.FS;
        }
        throw DbException.get(90060, method);
    }

    public String getUniqueId() {
        return this.uniqueId;
    }

    @Override
    public void run() {
        try {
            ServerSocket local;
            while (this.locked && this.fileName != null) {
                try {
                    if (!FileUtils.exists(this.fileName) || FileUtils.lastModified(this.fileName) != this.lastWrite) {
                        this.save();
                    }
                    Thread.sleep(this.sleep);
                }
                catch (InterruptedException | NullPointerException | OutOfMemoryError throwable) {
                }
                catch (Exception e) {
                    this.trace.debug(e, "watchdog");
                }
            }
            while ((local = this.serverSocket) != null) {
                try {
                    this.trace.debug("watchdog accept");
                    Socket s2 = local.accept();
                    s2.close();
                }
                catch (Exception e) {
                    this.trace.debug(e, "watchdog");
                }
            }
        }
        catch (Exception e) {
            this.trace.debug(e, "watchdog");
        }
        this.trace.debug("watchdog end");
    }
}

