#!/usr/bin/env bash
#
# Copyright (C) GridGain Systems. All Rights Reserved.
#  _________        _____ __________________        _____
#  __  ____/___________(_)______  /__  ____/______ ____(_)_______
#  _  / __  __  ___/__  / _  __  / _  / __  _  __ `/__  / __  __ \
#  / /_/ /  _  /    _  /  / /_/ /  / /_/ /  / /_/ / _  /  _  / / /
#  \____/   /_/     /_/   \_,__/   \____/   \__,_/  /_/   /_/ /_/
#

#
# SSL certificates generation.
#

#
# Preconditions:
#  1. If needed, install Open SSL (for example: "sudo apt-get install openssl")
#  2. If needed, install JDK 8 or newer. We need "keytool" from "JDK/bin".
#  3. Create "openssl.cnf" in some folder (for example: "/opt/openssl").
#     You may use "https://github.com/openssl/openssl/blob/master/apps/openssl.cnf" as template.
#  4. If needed, add "opensll" & "keytool" to PATH variable.
#
#  NOTE: In case of custom SERVER_DOMAIN_NAME you may need to tweak your "etc/hosts" file.
#

set -x

# Set Open SSL variables.
OPENSSL_CONF=/opt/openssl/openssl.cnf

# Certificates password.
PWD=123456

# Server.
SERVER_DOMAIN_NAME=localhost
SERVER_EMAIL=support@test.local

# Client.
CLIENT_DOMAIN_NAME=localhost
CLIENT_EMAIL=client@test.local

# Control Center.
CONTROL_DOMAIN_NAME=control.gridgain.com
CONTROL_EMAIL=control@gridgain.com

# Cleanup.
rm -vf server.*
rm -vf client.*
rm -vf ca.*
rm -vf control.*

# Generate server config.
cat << EOF > server.cnf
[req]
prompt                 = no
distinguished_name     = dn
req_extensions         = req_ext
[ dn ]
countryName            = RU
stateOrProvinceName    = Moscow
localityName           = Moscow
organizationName       = test
commonName             = ${SERVER_DOMAIN_NAME}
organizationalUnitName = IT
emailAddress           = ${SERVER_EMAIL}
[ req_ext ]
subjectAltName         = @alt_names
[ alt_names ]
DNS.1                  = ${SERVER_DOMAIN_NAME}
DNS.2                  = host.docker.internal
DNS.3                  = host.testcontainers.internal
EOF

# Generate client config.
cat << EOF > client.cnf
[req]
prompt                 = no
distinguished_name     = dn
req_extensions         = req_ext
[ dn ]
countryName            = RU
stateOrProvinceName    = Moscow
localityName           = Moscow
organizationName       = test
commonName             = ${CLIENT_DOMAIN_NAME}
organizationalUnitName = IT
emailAddress           = ${CLIENT_EMAIL}
[ req_ext ]
subjectAltName         = @alt_names
[ alt_names ]
DNS.1                  = ${CLIENT_DOMAIN_NAME}
DNS.2                  = host.docker.internal
DNS.3                  = host.testcontainers.internal
IP.1                   = 127.0.0.1
EOF

# Generate Control Center config.
cat << EOF > control.cnf
[req]
prompt                 = no
distinguished_name     = dn
req_extensions         = req_ext
[ dn ]
countryName            = RU
stateOrProvinceName    = Moscow
localityName           = Moscow
organizationName       = test
commonName             = ${CONTROL_DOMAIN_NAME}
organizationalUnitName = IT
emailAddress           = ${CONTROL_EMAIL}
[ req_ext ]
subjectAltName         = @alt_names
[ alt_names ]
DNS.1                  = ${CONTROL_DOMAIN_NAME}
EOF


# Generate certificates.
openssl genrsa -des3 -passout pass:${PWD} -out server.key 1024
openssl req -new -passin pass:${PWD} -key server.key -config server.cnf -out server.csr

openssl genrsa -des3 -passout pass:${PWD} -out server-outdated.key 1024
openssl req -new -passin pass:${PWD} -key server-outdated.key -config server.cnf -out server-outdated.csr

openssl req -new -newkey rsa:1024 -nodes -keyout ca.key -x509 -days 3650 -config server.cnf -out ca.crt
openssl req -new -newkey rsa:1024 -nodes -keyout ca-other.key -x509 -days 3650 -config server.cnf -out ca-other.crt

openssl x509 -req -days 3650 -in server.csr -CA ca.crt -CAkey ca.key -set_serial 01 -extensions req_ext -extfile server.cnf -out server.crt
openssl x509 -req -days 1 -in server-outdated.csr -CA ca.crt -CAkey ca.key -set_serial 02 -extensions req_ext -extfile server.cnf -out server-outdated.crt

openssl req -new -utf8 -nameopt multiline,utf8 -newkey rsa:1024 -nodes -keyout client.key -config client.cnf -out client.csr
openssl x509 -req -days 3650 -in client.csr -CA ca.crt -CAkey ca.key -set_serial 03 -extensions req_ext -extfile client.cnf -out client.crt

openssl req -new -sha256 -utf8 -nameopt multiline,utf8 -newkey rsa:1024 -nodes -keyout control.key -config control.cnf -out control.csr
openssl x509 -req -days 3650 -in control.csr -CA ca.crt -CAkey ca.key -set_serial 04 -out control.crt

openssl pkcs12 -export -in server.crt -inkey server.key -certfile ca.crt -out server.p12 -passin pass:${PWD} -passout pass:${PWD}
openssl pkcs12 -export -in server-outdated.crt -inkey server-outdated.key -certfile ca.crt -out server-outdated.p12 -passin pass:${PWD} -passout pass:${PWD}
openssl pkcs12 -export -in client.crt -inkey client.key -certfile ca.crt -out client.p12 -passout pass:${PWD}
openssl pkcs12 -export -in ca.crt -inkey ca.key -certfile ca.crt -out ca.p12 -passout pass:${PWD}
openssl pkcs12 -export -in ca-other.crt -inkey ca-other.key -certfile ca-other.crt -out ca-other.p12 -passout pass:${PWD}
openssl pkcs12 -export -in control.crt -inkey control.key -certfile ca.crt -out control.p12 -passin pass:${PWD} -passout pass:${PWD}

keytool -import -alias cacrt -file ca.crt -storetype PKCS12 -noprompt -storepass ${PWD} -keystore ca.p12
keytool -import -alias cacrt -file ca-other.crt -storetype PKCS12 -noprompt -storepass ${PWD} -keystore ca-other.p12

# Cleanup.
rm -vf *.cnf
rm -vf ca-other.crt
rm -vf server*.crt
rm -vf client.crt
rm -vf control.crt
rm -vf ca-other.key
rm -vf server*.key
rm -vf client.key
rm -vf control.key
rm -vf *.csr
